import logging
import requests

class WikiApi(object):
    def __init__(self,
                  wiki_url,
                  oauth_token):
        self.wiki_url = wiki_url
        self.oauth_token = oauth_token


    def auth_header(self):
        return {
            'Authorization': 'Bearer ' + self.oauth_token
        }


    def create_wiki(self, entity_path, grid_template):
        requests.delete(u'{}/frontend{}'.format(self.wiki_url, entity_path), headers=self.auth_header())

        resp = requests.post(
            '{}/frontend{}/.grid/clone'.format(self.wiki_url, grid_template),
            json={
                "destination": entity_path,
                "with_data": False
            },
            headers=self.auth_header()
        )

        logging.info(resp.json())

    def check_page_exists(self, entity_path):
        return requests.head(u'{}/frontend{}'.format(self.wiki_url, entity_path)).status_code == 200

    def create_page(self, entity_path, title, body, rewrite=True):
        page_exists = self.check_page_exists(entity_path)
        if not rewrite and page_exists:
            return

        if page_exists:
            self.delete_page(entity_path)

        resp = requests.post(u'{}/frontend{}'.format(self.wiki_url, entity_path),
            data= {"title": title, "body": body},
            headers=self.auth_header()
        )

        logging.info(resp.content)

    def delete_page(self, entity_path):
        json = requests.delete(u'{}/frontend{}'.format(self.wiki_url, entity_path)).json()
        logging.info(json)
        return ['data']['success']
        

    def rename(self, entity, entity_path, grid_template):
        resp = requests.get(u'{}/frontend{}/.grid'.format(self.wiki_url, entity_path), headers=self.auth_header())

        logging.info(resp.content)

        version = resp.json()["data"]["version"]

        resp2 = requests.post(
            u'{}/frontend{}/.grid/change'.format(self.wiki_url, entity_path),
            json={
                "version": version,
                "changes": [
                    {
                        "title_changed": {
                            "title": entity
                        }
                    }
                ]
            },
            headers=self.auth_header()
        )

        logging.info(resp2.content)


    def upload_csv(self, file_name, file_path):
        params = {
            'delimiter': ',',
            'quotechar': '"',
            'charset': 'utf-8',
            'omit_first': 'false'
        }
        multipart_form_data = {
            'file': (file_name, open(file_path, 'r')),

        }

        resp = requests.post('{}/frontend/.import/grid/upload'.format(self.wiki_url), files=multipart_form_data, headers=self.auth_header(), params=params)

        logging.info(resp.content)

        return resp.json()['data']['cache_key']

    def import_csv(self, entity_path, cache_key, columns):
        headers = {
            'Content-Type': 'application/json'
        }
        headers.update(self.auth_header())

        params = {
            'key': cache_key,
            'delimiter': ','
        }
        params.update(columns)

        resp = requests.post(u'{}/frontend{}/.grid/import'.format(self.wiki_url, entity_path), headers=headers, params=params)
        logging.info(resp.content)
