# coding=utf-8

import json
import logging
import os

import requests


class Generator(object):
    def __init__(self,
                 oauthToken,
                 solomonUrl,
                 projectId,
                 checkouterUrl,
                 groupBlackList,
                 channels,
                 paymentAlertsChannels,
                 tasksForPaymentAlerts,
                 periodMillis=300000,
                 delaySeconds=120):
        self.solomonUrl = solomonUrl
        self.projectId = projectId
        self.headers = {"Authorization": "OAuth " + oauthToken, "Content-Type": "application/json"}
        self.checkouterUrl = checkouterUrl
        self.periodMillis = periodMillis
        self.delaySeconds = delaySeconds
        self.groupBlackList = groupBlackList
        self.channels = channels
        self.paymentAlertsChannels = paymentAlertsChannels
        self.tasksForPaymentAlerts = tasksForPaymentAlerts

    def generate(self):
        groups = self.get_groups()

        logging.info("Loaded {} groups", len(groups))

        for group in groups:
            alert = self.create_alert(group)

            alertResponse = requests.get(self.solomonUrl + '/alerts/' + alert['id'], headers=self.headers)

            if alertResponse.status_code == requests.codes.ok:
                logging.info('alert %s found... update it', alert['id'])
                loaded = alertResponse.json()
                alert['version'] = loaded['version']
                p = requests.put(self.solomonUrl + '/alerts/' + alert['id'], headers=self.headers, json=alert)
                if p.status_code <> requests.codes.ok:
                    logging.info(p.content)
            else:
                logging.info('alert %s not found... post it', alert['id'])
                logging.debug('alert: %s', json.dumps(alert))
                p = requests.post(self.solomonUrl + '/alerts', headers=self.headers, json=alert)
                if p.status_code <> requests.codes.ok:
                    logging.info(p.content)

    def get_groups(self):
        return requests.get(self.checkouterUrl).json()

    def create_alert(self, group):
        group_name = group['groupName']
        description = u'Описание: ' + group['description'] + '. ' if len(group['description']) > 0 else ''
        task_name = u'Таска: ' + group['taskName'] if 'taskName' in group and len(group['taskName']) > 0 else ''
        threshold = group['threshold'] if 'threshold' in group else 0

        id = 'c-h-m-' + group_name

        program = u"""
let raw_data={{
    project='market-checkout',
    cluster='stable',
    service='market-checkouter',
    period='one_min',
    sensor='monitoring_error_count',
    group_name='{group}'
}};

let raw_data_size = count(raw_data);
let no_data = raw_data_size == 0;
alarm_if(no_data && last(alert_evaluation_history()) == 4);
no_data_if(no_data);

let threshold = {threshold};
let error_count = last(raw_data);
let description = error_count <= threshold ?
    "Все в порядке с мониторингом {group}" :
    "Обнаружено {{{{error_count}}}} ошибок в мониторинге {group}. Смотрите таблицу select * from monitorings where group_name = '{group}' для получения дополнительной информации. {description}{task_name}";

alarm_if(error_count > threshold);
        """.format(group=group_name, description=description, task_name=task_name, threshold=threshold)

        logging.info("program: %s", program)

        if group_name in self.groupBlackList:
            channels = []
        elif group_name in self.tasksForPaymentAlerts:
            channels = self.paymentAlertsChannels
        else:
            channels = self.channels
        logging.info('Selected channels: {}'.format(channels))

        return {
            "id": id,
            "projectId": self.projectId,
            "name": u'Мониторинг ' + group_name,
            "groupByLabels": [],
            "notificationChannels": channels,
            "annotations": {
                "group_name": group_name,
                "description": '{{expression.description}}',
                "raw_data_size": '{{expression.raw_data_size}}'
            },
            "periodMillis": self.periodMillis,
            "delaySeconds": self.delaySeconds,
            "type": {
                "expression": {
                    "program": program,
                    "checkExpression": "false"
                }
            }
        }


def main():
    logging.basicConfig(level=logging.DEBUG)

    oauthToken = os.environ['SOLOMON_OAUTH']
    projectId = 'market-checkouter-devel'
    solomonUrl = 'http://solomon.yandex.net/api/v2/projects/{}'.format(projectId)
    checkouterUrl = 'http://checkouter.tst.vs.market.yandex.net:39001/monitor/dev/groups'
    groupBlackList = set()

    generator = Generator(oauthToken, solomonUrl, projectId, checkouterUrl, groupBlackList)
    generator.generate()


if __name__ == '__main__':
    main()
