# coding=utf-8

import json
import logging
import os

import requests


class Generator(object):
    def __init__(self,
                 oauth_token,
                 solomon_url,
                 project_id,
                 checkouter_url,
                 channels,
                 payment_channels,
                 period_millis=300000,
                 delay_seconds=120):
        self.solomonUrl = solomon_url
        self.projectId = project_id
        self.headers = {"Authorization": "OAuth " + oauth_token, "Content-Type": "application/json"}
        self.checkouter_url = checkouter_url
        self.channels = channels
        self.payment_channels = payment_channels
        self.periodMillis = period_millis
        self.delaySeconds = delay_seconds

    def generate(self):
        self.generate_qc_error_count_alerts()

    def generate_qc_error_count_alerts(self):
        queued_call_types = self.get_queued_call_types()

        logging.info("Loaded {} groups", len(queued_call_types))

        infra_types = [qc for qc in queued_call_types if not qc['paymentTask']]
        payment_types = [qc for qc in queued_call_types if qc['paymentTask']]

        self.generate_alert('generated-queued-calls-count-infra',
                            lambda alert_id: self.create_qc_count_alert(alert_id, infra_types, self.channels))
        self.generate_alert('generated-queued-calls-count-fintech',
                            lambda alert_id: self.create_qc_count_alert(alert_id, payment_types, self.payment_channels))
        self.generate_alert('generated-queued-calls-errors-count-infra',
                            lambda alert_id: self.create_qc_error_alert(alert_id, infra_types, self.channels))
        self.generate_alert('generated-queued-calls-errors-count-fintech',
                            lambda alert_id: self.create_qc_error_alert(alert_id, payment_types, self.payment_channels))

    def get_queued_call_types(self):
        return requests.get(self.checkouter_url + "/queuedcalls/types").json()

    def generate_alert(self, alert_id, alert_generator):
        alert = alert_generator(alert_id)
        logging.debug('alert: %s', json.dumps(alert))
        old_alert = requests.get('{}/alerts/{}'.format(self.solomonUrl, alert_id), headers=self.headers)
        if old_alert.status_code == requests.codes.ok:
            logging.info('alert %s found... update it', alert_id)
            alert['version'] = old_alert.json()['version']
            p = requests.put('{}/alerts/{}'.format(self.solomonUrl, alert_id), headers=self.headers, json=alert)
            if p.status_code != requests.codes.ok:
                raise Exception("Exception while updating alert...\n" + p.content)
        else:
            logging.info('alert %s not found... post it', alert_id)
            p = requests.post(self.solomonUrl + '/alerts', headers=self.headers, json=alert)
            if p.status_code != requests.codes.ok:
                raise Exception("Exception while creating alert...\n" + p.content)

    def create_qc_count_alert(self, alert_id, qc_types, channels):
        qc_names = '|'.join((qc['name'] for qc in qc_types))

        program = u"""
let raw_data={{
    cluster='stable',
    project='market-checkout',
    service='market-checkouter',
    host='-',
    key='queued_calls_count',
    subkey='{qc_names}'}};

let raw_data_size = count(raw_data);
let no_data = raw_data_size == 0;
alarm_if(no_data && last(alert_evaluation_history()) == 4);
no_data_if(no_data);

let threshold = 1000;
let qc_type = get_label(raw_data, "subkey");
// Добавляем кастомные пороги
let threshold = qc_type == "TRACK_START_TRACKING" ? 100000 : threshold;
let above_threshold = count(drop_below(raw_data, threshold));
let description = 'Размер очереди превысил порог в {{{{threshold}}}} {{{{above_threshold}}}} раз за последние 5 минут. Смотрите в таблице queued_calls where processed_at is not null что за вызовы и почему их так много';
alarm_if(above_threshold > 0);

let description = 'Всё хорошо с обработкой очереди';
""".format(qc_names=qc_names)

        return {
            "id": alert_id,
            "projectId": self.projectId,
            "name": alert_id,
            "groupByLabels": ['subkey'],
            "channels": [
                {
                    "id": channel,
                    "config": {
                        "notifyAboutStatuses": [
                            "ALARM"
                        ],
                        "repeatDelaySecs": 86400
                    }
                } for channel in channels
            ],
            "annotations": {
                "queued_call_type": '{{labels.subkey}}',
                "description": '{{expression.description}}',
                "raw_data_size": '{{expression.raw_data_size}}'
            },
            "periodMillis": self.periodMillis,
            "delaySeconds": self.delaySeconds,
            "type": {
                "expression": {
                    "program": program,
                    "checkExpression": "false"
                }
            }
        }

    def create_qc_error_alert(self, alert_id, qc_types, channels):
        qc_names = '|'.join((qc['name'] for qc in qc_types))

        program = u"""
let raw_data={{
    cluster='stable',
    project='market-checkout',
    service='market-checkouter',
    host='-',
    key='queued_calls_errors',
    subkey='{qc_names}'}};

let raw_data_size = count(raw_data);
let no_data = raw_data_size == 0;
alarm_if(no_data && last(alert_evaluation_history()) == 4);
no_data_if(no_data);

let above_threshold = count(drop_below(raw_data, 200));
let description = 'Количество ошибок превысило порог в 200 {{{{above_threshold}}}} раз за последние 5 минут. Смотрите в таблице queued_calls where processed_at is not null and last_try_error is not null что за ошибки и по каким типам вызовов. Для разбора из этой таблицы полезны будут поля:\n trace_id - трейс обработки последнего вызова (по нему можно найти хост и логи) \nlast_try_error - собственно текст исключения и кусок стектрейса';
alarm_if(above_threshold > 0);

let description = 'Всё хорошо с обработкой очереди';
""".format(qc_names=qc_names)

        return {
            "id": alert_id,
            "projectId": self.projectId,
            "name": alert_id,
            "groupByLabels": ['subkey'],
            "channels": [
                {
                    "id": channel,
                    "config": {
                        "notifyAboutStatuses": [
                            "ALARM"
                        ],
                        "repeatDelaySecs": 86400
                    }
                } for channel in channels
            ],
            "annotations": {
                "queued_call_type": '{{labels.subkey}}',
                "description": '{{expression.description}}',
                "raw_data_size": '{{expression.raw_data_size}}'
            },
            "periodMillis": self.periodMillis,
            "delaySeconds": self.delaySeconds,
            "type": {
                "expression": {
                    "program": program,
                    "checkExpression": "false"
                }
            }
        }


def main():
    logging.basicConfig(level=logging.DEBUG)

    oauthToken = os.environ['SOLOMON_OAUTH']
    projectId = 'market-checkout'
    solomonUrl = 'http://solomon.yandex.net/api/v2/projects/{}'.format(projectId)
    checkouterUrl = 'http://checkouter.tst.vs.market.yandex.net:39001'

    generator = Generator(oauthToken, solomonUrl, projectId, checkouterUrl, ['telegram'],
                          ['telegram_market_payment_alerts'])
    generator.generate()


if __name__ == '__main__':
    main()
