# coding=utf-8
import codecs
import logging
import os
from collections import OrderedDict

import utils
import constants


class EntityBuilder:

    def __init__(self,
                oauth_token,
                wiki_url = 'https://wiki-api.yandex-team.ru/_api',
                root_page = u'/users/timursha/autogen'):
        self.oauth_token = oauth_token
        self.wiki_url = wiki_url
        self.root_page = root_page
        self.grid_template = u'/users/timursha/autogen/grid'

    def auth_header(self):
        return {
            'Authorization': 'Bearer ' + self.oauth_token
        }

    def get_entity_path(self, entity):
        # replacing «»
        return self.root_page + '/' + utils.escape_entity(entity)

    def process_definitions(self, swagger):
        if "definitions" not in swagger:
            return

        definitions = swagger["definitions"]

        logging.info('loaded %d definitions', len(definitions))

        for entity, definition in definitions.iteritems():
            logging.info(u'Processing entity = ' + entity)

            in_properties = filter_properties(definition, "IN")
            out_properties = filter_properties(definition, "OUT")

            # process in entity
            in_entity = entity + 'Request'
            self.process_definition(in_entity, definition, in_properties)

            # process entity
            self.process_definition(entity, definition, out_properties)

    def process_definition(self, entity, definition, properties):
        if len(properties) == 0:
            return

        required = definition['required'] if 'required' in definition else []

        self.write_csv(entity, properties, required)

        self.create_wiki(entity)
        self.rename(entity)

        cache_key = self.upload_csv(entity)
        self.import_csv(entity, cache_key)

    def create_wiki(self, entity):
        utils.requests_retry_session().delete(
            u'{}/frontend{}'.format(self.wiki_url, self.get_entity_path(entity)),
            headers=self.auth_header()
        )

        resp = utils.requests_retry_session().post(
            '{}/frontend{}/.grid/clone'.format(self.wiki_url, self.grid_template),
            json={
                "destination": self.get_entity_path(entity),
                "with_data": False
            },
            headers=self.auth_header()
        )

        logging.info(resp.json())

    def write_csv(self, entity, properties, required):
        if not os.path.exists("csv"):
            os.makedirs("csv")

        file_name = 'csv/' + entity + '.csv'
        with codecs.open(file_name, 'w', 'utf-8') as f:
            for prop, property_definition in properties.iteritems():
                deprecated = property_definition["deprecated"] == "true" if "deprecated" in property_definition else False

                property_type = utils.match_type(property_definition)
                property_required = "true" if prop in required else ""
                property_min = ''
                property_max = ''
                property_format = utils.get_property_format(property_definition, root_page=self.root_page)
                property_description = property_definition['description'] if 'description' in property_definition else ''
                property_default = property_definition['default'] if 'default' in property_definition else ''

                line = u"\"{}\",\"{}\",\"{}\",\"{}\",\"{}\",\"{}\",\"{}\",\"{}\",\"{}\",\"{}\"\n".format(
                    prop if not deprecated else "--{}--".format(prop),
                    property_type[0],
                    "true" if property_type[1] else "",
                    '',
                    property_required,
                    property_min,
                    property_max,
                    property_format,
                    property_description.replace("\"", "\"\""),
                    property_default
                )

                f.write(line)
        return file_name

    def rename(self, entity):
        resp = utils.requests_retry_session().get(
            u'{}/frontend{}/.grid'.format(self.wiki_url, self.get_entity_path(entity)),
            headers=self.auth_header()
        )

        print resp.content

        version = resp.json()["data"]["version"]

        resp2 = utils.requests_retry_session().post(
            u'{}/frontend{}/.grid/change'.format(self.wiki_url, self.get_entity_path(entity)),
            json={
                "version": version,
                "changes": [
                    {
                        "title_changed": {
                            "title": entity
                        }
                    }
                ]
            },
            headers=self.auth_header()
        )

        print resp2.content

    def upload_csv(self, entity):
        params = {
            'delimiter': ',',
            'quotechar': '"',
            'charset': 'utf-8',
            'omit_first': 'false'
        }
        multipart_form_data = {
            'file': (entity + '.csv', open('csv/' + entity + '.csv', 'r')),

        }

        resp = utils.requests_retry_session().post(
            '{}/frontend/.import/grid/upload'.format(self.wiki_url),
            files=multipart_form_data,
            headers=self.auth_header(),
            params=params
        )

        print resp.content

        return resp.json()['data']['cache_key']

    def import_csv(self, entity, cache_key):
        headers = {}
        headers['Authorization'] = self.auth_header()['Authorization']
        headers['Content-Type'] = 'application/json'
        resp = utils.requests_retry_session().post(
            constants.IMPORT_CSV_URL_PATTERN.format(self.wiki_url, self.get_entity_path(entity), cache_key),
            headers=headers
        )

        print resp.content

def get_directions(property_definition):
    if "directions" not in property_definition:
        return ["IN", "OUT"]

    return property_definition["directions"].split(",")

def filter_properties(defintion, direction):
    if 'properties' not in defintion:
        return {}

    properties = defintion['properties']
    result = OrderedDict()
    for prop, property_definition in properties.iteritems():
        directons = get_directions(property_definition)
        if direction not in directons:
            continue

        result[prop] = property_definition

    return result
