# coding=utf-8
import codecs
import logging
import os
from collections import OrderedDict

import utils


class EnumBuilder():
    def __init__(self,
                oauth_token,
                wiki_url = 'https://wiki-api.yandex-team.ru/_api',
                root_page =  u'/users/timursha/autogen'):
        self.oauth_token = oauth_token
        self.wiki_url = wiki_url
        self.enum_grid_template = '/users/timursha/autogen/grid_enum2'
        self.root_page = root_page

    def auth_header(self):
        return {
            'Authorization': 'Bearer ' + self.oauth_token
        }

    def get_entity_path(self, entity):
        # replacing «»
        return self.root_page + '/' + utils.escape_entity(entity)

    def collect_enums(self, swagger):
        if "definitions" not in swagger:
            return {}

        definitions = swagger['definitions']
        paths = swagger['paths']

        enums = {}

        for entity, definition in definitions.iteritems():

            if 'properties' not in definition:
                continue

            for prop, property_definition in definition['properties'].iteritems():
                enum = detect_enum(property_definition)
                if enum is None:
                    continue

                enums[enum["enumClass"]] = enum['enumDescriptions']

        for path, path_definition in paths.iteritems():
            for method, method_definition in path_definition.iteritems():
                if 'parameters' not in method_definition:
                    continue

                for parameter_definition in method_definition['parameters']:
                    enum = detect_enum(parameter_definition)

                    if enum is None:
                        continue

                    enums[enum["enumClass"]] = enum["enumDescriptions"]

        return enums

    def process_enums(self, swagger):
        enums = self.collect_enums(swagger)

        for enumClass, enum in enums.iteritems():
            logging.info("{}: {}".format(enumClass, enum))
            self.process_enum(enumClass, enum)

    def process_enum(self, enumClass, enum):
        self.create_enum_wiki(enumClass)
        self.rename(enumClass)
        file_name = self.write_enum_csv(enumClass, enum)
        if file_name is None:
            return
        cache_key = self.upload_csv(enumClass)
        self.import_enum_csv(enumClass, cache_key)

    def create_enum_wiki(self, enum):
        utils.requests_retry_session().delete(
            u'{}/frontend{}'.format(self.wiki_url, self.get_entity_path(enum)),
            headers=self.auth_header()
        )

        resp = utils.requests_retry_session().post(
            '{}/frontend{}/.grid/clone'.format(self.wiki_url, self.enum_grid_template),
            json={
                "destination": self.get_entity_path(enum),
                "with_data": False
            },
            headers=self.auth_header()
        )

        logging.info(resp.json())

    def rename(self, entity):
        resp = utils.requests_retry_session().get(
            u'{}/frontend{}/.grid'.format(self.wiki_url, self.get_entity_path(entity)),
            headers=self.auth_header()
        )

        print resp.content

        version = resp.json()["data"]["version"]

        resp2 = utils.requests_retry_session().post(
            u'{}/frontend{}/.grid/change'.format(self.wiki_url, self.get_entity_path(entity)),
            json={
                "version": version,
                "changes": [
                    {
                        "title_changed": {
                            "title": entity
                        }
                    }
                ]
            },
            headers=self.auth_header()
        )

        print resp2.content

    def write_enum_csv(self, enumClass, enum):
        if len(enum) == 0:
            return None

        if not os.path.exists("csv"):
            os.makedirs("csv")

        file_name = 'csv/' + enumClass + '.csv'
        with codecs.open(file_name, 'w', 'utf-8') as f:
            for enumDescription in enum:
                enumValue = enumDescription['value']
                enumName = ''
                enumId = ''
                enumNote = ''
                for enumEntry in enumValue:
                    if (enumEntry['name'] == 'name'):
                        enumName = enumEntry['value']
                    elif enumEntry['name'] == 'id':
                        enumId = enumEntry['value']
                    elif enumEntry['name'] == 'description':
                        enumNote = enumEntry['value']

                line = u"\"{}\",\"{}\",\"{}\"\n".format(
                    enumName,
                    enumId,
                    enumNote
                )

                f.write(line)
        return file_name

    def upload_csv(self, entity):
        params = {
            'delimiter': ',',
            'quotechar': '"',
            'charset': 'utf-8',
            'omit_first': 'false'
        }
        multipart_form_data = {
            'file': (entity + '.csv', open('csv/' + entity + '.csv', 'r')),

        }

        resp = utils.requests_retry_session().post(
            '{}/frontend/.import/grid/upload'.format(self.wiki_url),
            files=multipart_form_data,
            headers=self.auth_header(),
            params=params
        )

        print resp.content

        return resp.json()['data']['cache_key']

    def import_enum_csv(self, entity, cache_key):
        headers = {
            "Authorization": self.auth_header()["Authorization"],
            "Content-Type": "application/json"
        }

        resp = utils.requests_retry_session().post(
            u'{}/frontend{}/.grid/import'.format(self.wiki_url, self.get_entity_path(entity)),
            params={
                "key": cache_key,
                "icolumn_0_to": "101",
                "icolumn_1_to": "100",
                "icolumn_2_to": "102",
                "delimiter": ","
            },
            headers=headers
        )

        print resp.content


def detect_enum(definition):
    if 'enumDescriptions' not in definition:
        return None

    return {
        "enumClass": definition['enumClass'],
        "enumDescriptions": definition['enumDescriptions']
    }


def main():
    resp = utils.requests_retry_session().get("http://gravicapa01ht.market.yandex.net:39001/v2/api-docs")
    swagger = resp.json(object_pairs_hook=OrderedDict)

    builder = EnumBuilder(os.environ['oauth_token'])
    builder.process_enums(swagger)
