# coding=utf-8

import codecs
import logging
import os
from collections import OrderedDict

import utils
import constants

logging.basicConfig(level=logging.DEBUG, filename="wiki.log")
log = logging.getLogger(__name__)


class HeadersBuilder(object):
    def __init__(self,
                 oauth_token,
                 wiki_url = 'https://wiki-api.yandex-team.ru/_api',
                 root_page = u'/users/timursha/autogen-headers',
                 grid_template = '/users/timursha/autogen/grid',
                 root_entity_page = u'/users/timursha/autogen'):
        self.oauth_token = oauth_token
        self.wiki_url = wiki_url
        self.root_page = root_page
        self.grid_template = grid_template
        self.root_entity_page = root_entity_page

    def auth_header(self):
        return {
            'Authorization': 'Bearer ' + self.oauth_token
        }

    def get_entity_path(self, operationId):
        return self.root_page + '/' + operationId

    def process_path(self, path, path_definition):
        for method, definition in path_definition.iteritems():
            if "operationId" not in definition:
                continue

            operationId = definition["operationId"]

            print "operationId = {}".format(operationId)

            filename = self.process_path_with_method(operationId, definition)
            if filename is None:
                return
            self.create_wiki(operationId)
            self.rename(operationId)
            cache_key = self.upload_csv(operationId)
            self.import_csv(operationId, cache_key)

    def process_path_with_method(self, operationId, definition):
        if "parameters" not in definition:
            return

        parameters_definition = definition['parameters']
        parameters = []

        for parameter_definition in parameters_definition:
            if parameter_definition["in"] <> "header":
                # пропускаем тело
                continue

            parameters.append(parameter_definition)

        return self.write_csv(operationId, parameters)

    def create_wiki(self, entity):
        utils.requests_retry_session().delete(
            u'{}/frontend{}'.format(self.wiki_url, self.get_entity_path(entity)),
            headers=self.auth_header()
        )

        resp = utils.requests_retry_session().post(
            '{}/frontend{}/.grid/clone'.format(self.wiki_url, self.grid_template),
            json={
                "destination": self.get_entity_path(entity),
                "with_data": False
            },
            headers=self.auth_header()
        )

        log.info(resp.json())

    def rename(self, entity):
        resp = utils.requests_retry_session().get(
            u'{}/frontend{}/.grid'.format(self.wiki_url, self.get_entity_path(entity)),
            headers=self.auth_header()
        )

        print resp.content

        version = resp.json()["data"]["version"]

        resp2 = utils.requests_retry_session().post(
            u'{}/frontend{}/.grid/change'.format(self.wiki_url, self.get_entity_path(entity)),
            json={
                "version": version,
                "changes": [
                    {
                        "title_changed": {
                            "title": entity
                        }
                    }
                ]
            },
            headers=self.auth_header()
        )

        print resp2.content

    def write_csv(self, operationId, parameters):
        if len(parameters) == 0:
            return

        if not os.path.exists("headers_csv"):
            os.makedirs("headers_csv")

        file_name = 'headers_csv/' + operationId + '.csv'
        with codecs.open(file_name, 'w', 'utf-8') as f:
            for parameter in parameters:
                parameter_name = parameter["name"]
                parameter_type, is_array = utils.match_type(parameter)
                description = parameter["description"] if "description" in parameter and parameter["description"] != parameter_name else ""
                description = description.replace("\"", "\"\"")
                format = utils.get_property_format(parameter, root_page=self.root_entity_page)

                f.write(u"\"{}\",\"{}\",\"{}\",\"{}\",\"{}\",\"{}\",\"{}\",\"{}\",\"{}\",\"{}\"\n".format(
                    parameter_name,
                    parameter_type,
                    "true" if is_array else "",
                    get_byte_size(parameter),
                    is_required(parameter),
                    # min
                    "",
                    # max
                    "",
                    # format
                    format,
                    # description
                    description,
                    # default
                    ""
                ))

        return file_name

    def import_csv(self, entity, cache_key):
        headers = {}
        headers['Authorization'] = self.auth_header()['Authorization']
        headers['Content-Type'] = 'application/json'
        resp = utils.requests_retry_session().post(
            constants.IMPORT_CSV_URL_PATTERN.format(self.wiki_url, self.get_entity_path(entity), cache_key),
            headers=headers
        )

        print resp.content

    def upload_csv(self, entity):
        params = {
            'delimiter': ',',
            'quotechar': '"',
            'charset': 'utf-8',
            'omit_first': 'false'
        }
        multipart_form_data = {
            'file': (entity + '.csv', open('headers_csv/' + entity + '.csv', 'r')),
        }

        resp = utils.requests_retry_session().post(
            '{}/frontend/.import/grid/upload'.format(self.wiki_url),
            files=multipart_form_data,
            headers=self.auth_header(),
            params=params
        )

        print resp.content

        return resp.json()['data']['cache_key']

    def process_definition(self, swagger):
        if "paths" not in swagger:
            return

        for path, path_definition in swagger["paths"].iteritems():
            self.process_path(path, path_definition)


def get_byte_size(parameter):
    if "type" in parameter and parameter["type"] == "array":
        return get_byte_size(parameter["items"])

    if "format" not in parameter:
        return ""

    format = parameter["format"]
    if "int32" == format:
        return "4"
    elif "int64" == format:
        return "8"
    else:
        return ""


def is_required(parameter):
    return "true" if "required" in parameter and parameter["required"] else ""


def main():
    resp = utils.requests_retry_session().get("http://gravicapa01ht.market.yandex.net:39001/v2/api-docs")

    swagger = resp.json(object_pairs_hook=OrderedDict)

    OAUTH_TOKEN = os.environ['oauth_token']

    builder = HeadersBuilder(OAUTH_TOKEN)
    builder.process_definition(swagger)
