#! /usr/bin/env python
# ya make --target-platform=default-linux-x86_64
# https://sandbox.yandex-team.ru/oauth/
# ./MarketCheckouterLoadCalculateRps upload --verbose --owner MARKET --attr release=stable --attr ttl=inf --attr task_type=MARKET_CHECKOUTER_LOAD_CALCULATE_RPS

import codecs
import csv
import logging as log

from sandbox import sdk2
from sandbox.projects.common.binary_task import deprecated as binary_task

from sandbox.projects.market.checkout.helpers import RTC
from sandbox.projects.market.checkout.helpers import YQL
from sandbox.projects.market.checkout.resources import CheckouterLoadEndpointsRps

ENDPOINTS_RPS = [
    ['POST cart BLUE', 0.09956551203267389],
    ['POST checkout BLUE', 0.003733880969208817],
    ['POST actualize BLUE', 0.010703842624190942],
    ['GET orders_by-uid_userId BLUE', 0.04424103413955259],
    ['GET orders_by-uid_userId_recent BLUE', 0.20353454870350682],
    ['GET orders_events BLUE', 0.019767444679765152],
    ['GET orders_orderId BLUE', 0.09927759101637937],
    ['GET orders_orderId_receipts BLUE', 0.037760538212271134],
    ['GET orders_events_by-order-id BLUE', 0.07904341121546311],
    ['POST notify-tracks BLUE', 0.035059687542051625],
    ['POST offers_categorize BLUE', 0.053427836082625454],
    ['POST auth BLUE', 0.1328109940973161],
    ['POST get-orders BLUE', 0.16693024067772363],
    ['POST orders_orderId_status BLUE', 0.014143438007271369]
]


class MarketCheckouterLoadCalculateRps(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        yql_token_owner = sdk2.parameters.String('YQL token OWNER in Vault', default_value='zomb-emily')
        yql_token_name = sdk2.parameters.String('YQL token NAME in Vault', default_value='YQL_TOKEN')
        shoot_only_archive_api = sdk2.parameters.Bool('Shoot only at the archive API', default_value=False)
        use_archive_api_statistics = sdk2.parameters.Bool('Use archive API statistics', default_value=True)

        with sdk2.parameters.Output:
            prod_to_load_instance_count_ratio = sdk2.parameters.Float('prod-to-load instance count ratio',
                                                                      default_value='')
            prod_to_load_instance_power_ratio = sdk2.parameters.Float('prod-to-load instance power ratio',
                                                                      default_value=1)
            endpoints_rps = sdk2.parameters.String('Endpoints and RPS (csv)', default_value='')

    def on_execute(self):
        log.info('Starting on_execute')
        binary_task.LastBinaryTaskRelease.on_execute(self)
        yql_token = sdk2.Vault.data(self.Parameters.yql_token_owner, self.Parameters.yql_token_name)
        rps_query_resource_path = 'sandbox/projects/market/checkout/MarketCheckouterLoadCalculateRps/rps_query.sql'
        current_rps_total, rps_ratio = self.current_prod_rps_and_rps_ratio(rps_query_resource_path, yql_token)
        log.info('Total current checkouter RPS: ' + str(current_rps_total))
        log.info('RPS ratio result: ' + str(rps_ratio))
        instance_count_ratio = self.get_prod_to_load_instance_count_ratio()
        self.Parameters.prod_to_load_instance_count_ratio = instance_count_ratio
        self.publish_output_resource(rps_ratio)

    def current_prod_rps_and_rps_ratio(self, rps_query_resource_path, yql_token):
        rows = YQL.query_clickhouse(yql_token, rps_query_resource_path)
        log.debug('got rps rows: {}'.format(rows))
        sum_rps = sum(map(lambda r: float(r[3]), rows))
        sum_parts = 0
        log.info('current RPS total:' + str(sum_rps))
        rps_ratio = []
        for row in rows:
            http_method = row[0]
            method = row[1]
            archived = row[2]
            rps = float(row[3])

            rps_part = rps / sum_rps
            sum_parts += rps_part

            use_archive_api_statistics = self.Parameters.use_archive_api_statistics
            shoot_only_archive_api = self.Parameters.shoot_only_archive_api

            if use_archive_api_statistics:
                if shoot_only_archive_api and archived == 'false':
                    continue

                rps_ratio.append([http_method, method, archived, rps_part])
            elif archived == 'true':
                continue
            else:
                rps_ratio.append([http_method, method, 'true', rps_part * 0.3])
                if shoot_only_archive_api:
                    continue
                rps_ratio.append([http_method, method, archived, rps_part])

        log.debug('sum of all ratios:' + str(sum_parts))
        return sum_rps, rps_ratio

    def get_prod_to_load_instance_count_ratio(self):
        prod_services = RTC.get_services(category_prefix='market_checkout_checkouter_production',
                                         service_prefix='production_market_checkouter')
        dc_minus_1_prod_instance_count = \
            RTC.get_dc_minus_1_prod_instance_count(services=prod_services,
                                                   category_prefix='market_checkout_checkouter_production')
        log.info('DC-1 PROD instance count: {}'.format(dc_minus_1_prod_instance_count))
        return dc_minus_1_prod_instance_count

    def publish_output_resource(self, rps_config):
        data = sdk2.ResourceData(CheckouterLoadEndpointsRps(self, 'Checkouter load endpoints RPS', 'endpoints_rps.csv'))
        with codecs.open('endpoints_rps.csv', 'w', encoding='utf-8') as f:
            writer = csv.writer(f)
            for endpoint_rps in rps_config:
                # http_method,method,archived,rps
                writer.writerow([endpoint_rps[0]] + [endpoint_rps[1]] + [endpoint_rps[2]] + [str(endpoint_rps[3])])
        with open('endpoints_rps.csv', 'r') as f:
            self.Parameters.endpoints_rps = f.read()
        data.ready()
