# -*- coding: utf8 -*-
# ! /usr/bin/env python
# ya make --target-platform=default-linux-x86_64
# ./MarketCheckouterLoadGenerateAmmoFullLoad upload --verbose --owner MARKET --attr release=stable --attr ttl=inf --attr task_type=MARKET_CHECKOUTER_LOAD_GENERATE_AMMO_FULL_LOAD

import codecs
import csv
import logging as log
import random

from sandbox import sdk2
from sandbox.projects.common.binary_task import deprecated as binary_task

from generate import (
    cart,
    multicart_actualize,
    checkout,
    actualize,
    categorize,
    order_events,
    orders_by_id,
    orders_by_id_receipts,
    orders_by_uid,
    orders_by_uid_sort_by_importance,
    orders_by_uid_recent,
    orders_by_uid_recent_sort_by_importance,
    order_events_by_order_id,
    notify_tracks,
    auth,
    get_orders_user,
    orders_status,
    orders_orderid_events,
    pushapi_settings_shopid,
    orders_options_availabilities,
    orders_orderid_receipts_receiptid_pdf,
    orders_orderid_warranty,
    payments_paymentid,
    orders_orderid_payments,
    shops_datasourceid
)
from sandbox.projects.market.checkout.helpers.get_delivery_id import get_delivery_id
from sandbox.projects.market.checkout.helpers.get_offer import get_offer
from sandbox.projects.market.checkout.resources import CheckouterLoadAmmo
from sandbox.projects.market.checkout.resources import CheckouterLoadConfig
from sandbox.projects.market.checkout.resources import CheckouterLoadEndpointsRps

WRITE_METHODS = ["checkout", "notify_tracks", "orders_status"]
DEFAULT_AUTOSTOP_CONFIG = 'quantile(99,2500,3s) http(4xx,30%,10) http(50x,10%,10) http(51x,10%,10) net(xx,10%,10)'


class MarketCheckouterLoadGenerateAmmoFullLoad(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        imbalance_shooting = sdk2.parameters.Bool(
            'Shoot until service is unstable?',
            description='When checked "line" tank rps schedule will be used until service fails. '
                        'When unchecked then rps will be constant ( "const" schedule )', default=True
        )
        with imbalance_shooting.value[False]:
            test_duration_minutes = sdk2.parameters.Integer('Test duration minutes', default=10)
            total_rps = sdk2.parameters.Integer('Total rps', default=8)

        with imbalance_shooting.value[True]:
            start_rps = sdk2.parameters.Integer('Start RPS', default=5)
            target_rps = sdk2.parameters.Integer('Target RPS', default=50)
            rps_increase_step = sdk2.parameters.Integer('RPS increase step', default=1)
            rps_step_duration_seconds = sdk2.parameters.Integer('RPS step duration seconds', default=100)
            autostop_config = sdk2.parameters.String('Shooting autostop config', default=DEFAULT_AUTOSTOP_CONFIG)

        checkouter_url = sdk2.parameters.String('Checkouter url',
                                                default='http://checkouter.load.vs.market.yandex.net:39001')
        report_url = sdk2.parameters.String('Report url',
                                            default='http://warehouse-report.blue.tst.vs.market.yandex.net:17051')
        stocks_url = sdk2.parameters.String('Stocks url', default='https://bos.tst.vs.market.yandex.net/stocks')
        loyalty_url = sdk2.parameters.String('Loyalty url',
                                             default='http://market-loyalty.tst.vs.market.yandex.net:35815')

        available_promocodes = sdk2.parameters.String('Available promocodes',
                                                      default='COUPON_FOR_LOAD2,COUPON_FOR_LOAD,COUPON_FOR_LOAD3')
        shop_id = sdk2.parameters.Integer('Shop id', default=431782)
        coins_promo_id = sdk2.parameters.Integer('Coins promo id', default=10432)
        last_event_id_from = sdk2.parameters.Integer('Min eventId for last event id query', default=119452048)
        last_event_id_range = sdk2.parameters.Integer('Max eventId for last event id query', default=119462048)
        datafeed_id = sdk2.parameters.Integer('Datafeed id', default=200344277)

        delivery_type = sdk2.parameters.String('delivery_type', default='PICKUP')

        latest_order_from = sdk2.parameters.Integer('latest_order_from', default=7529713)
        latest_order_to = sdk2.parameters.Integer('latest_order_to', default=7530752)
        endpoints_rps_resource = sdk2.parameters.Resource('Endpoints RPS resource',
                                                          resource_type=CheckouterLoadEndpointsRps)
        custom_endpoints_rps = sdk2.parameters.String('Custom endpoints RPS', multiline=True)
        skip_options = sdk2.parameters.Bool('Skip getting options', default_value=False)
        coins_disabled = sdk2.parameters.Bool('Coins disabled', default_value=False)

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        self.init()
        self.generate_ammo()
        self.generate_config()

    def init(self):
        self.supported_methods = {
            ('POST', 'cart'): cart,
            ('POST', 'v2/multicart/actualize'): multicart_actualize,
            ('POST', 'actualize'): actualize,
            ('POST', 'offers_categorize'): categorize,
            ('POST', 'auth'): auth,
            ('POST', 'checkout'): checkout,
            ('POST', 'notify-tracks'): notify_tracks,
            ('POST', 'orders_orderId_status'): orders_status,
            ('POST', 'pushapi-settings_shopId'): pushapi_settings_shopid,
            ('POST', 'get-orders'): get_orders_user,
            ('GET', 'orders_by-uid_userId_recent'): orders_by_uid_recent,
            ('GET', 'orders_by-uid_userId_recent_sort_by_importance'): orders_by_uid_recent_sort_by_importance,
            ('GET', 'orders_options-availabilities'): orders_options_availabilities,
            ('GET', 'pushapi-settings_shopId'): pushapi_settings_shopid,
            ('GET', 'shops_datasourceId'): shops_datasourceid,
            ('GET', 'orders_orderId_receipts_receiptId_pdf'): orders_orderid_receipts_receiptid_pdf,
            ('GET', 'orders_orderId_warranty'): orders_orderid_warranty,
            ('GET', 'payments_paymentId'): payments_paymentid,
            ('GET', 'orders_orderId_payments'): orders_orderid_payments,
            ('GET', 'orders_orderId_receipts'): orders_by_id_receipts,
            ('GET', 'orders_events_by-order-id'): order_events_by_order_id,
            ('GET', 'orders_orderId_events'): orders_orderid_events,
            ('GET', 'orders_events'): order_events,
            ('GET', 'orders_orderId'): orders_by_id,
            ('GET', 'orders_by-uid_userId'): orders_by_uid,
            ('GET', 'orders_by-uid_userId_sort_by_importance'): orders_by_uid_sort_by_importance
        }

    def generate_ammo(self):
        ammo_meta = self.create_ammo_metadata()
        options = {}
        if not self.Parameters.skip_options:
            options = self.create_options()
        options['coins_disabled'] = bool(self.Parameters.coins_disabled)
        log.info('generating ammo')
        max_rounds = 1000000
        if self.Parameters.imbalance_shooting:
            total_rounds = min(max_rounds, self.calculate_total_rounds_for_imbalance_shooting())
        else:
            total_rounds = min(max_rounds, self.Parameters.total_rps * self.Parameters.test_duration_minutes * 60)
        log.info('total rounds: ' + str(total_rounds))
        ammos = []
        total_count = 0
        for meta_line in ammo_meta:
            http_method = meta_line["http_method"]
            method = meta_line["method"]
            generate_function = meta_line["generator"]
            archived = meta_line["archived"]
            rps_multiplier = float(meta_line["rps_multiplier"])

            count = max(int(rps_multiplier * total_rounds), 1)
            log.info('count for {} {} {} = {}'.format(http_method, method, archived, count))
            total_count += count

            if generate_function.is_archive_api:
                ammos.extend(generate_function(options, count, archived))
            else:
                ammos.extend(generate_function(options, count))
        log.info('total request count: ' + str(total_count))

        random.shuffle(ammos)

        data = sdk2.ResourceData(CheckouterLoadAmmo(self, 'Checkouter load ammo', 'ammo.txt'))
        with codecs.open('ammo.txt', 'w', encoding='utf-8') as f:
            for ammo in ammos:
                f.write(ammo)
        data.ready()

    def calculate_total_rounds_for_imbalance_shooting(self):
        total_rounds = 0
        rps = self.Parameters.start_rps
        while rps <= self.Parameters.target_rps:
            total_rounds += rps * self.Parameters.rps_step_duration_seconds
            rps += self.Parameters.rps_increase_step
        return total_rounds

    def generate_config(self):
        config_file_name = 'check.yaml'
        config_resource = sdk2.ResourceData(CheckouterLoadConfig(self, 'Checkouter load config', config_file_name))

        config = self.read_shooting_config_template()
        if self.Parameters.imbalance_shooting:
            config = self.generate_config_lines_for_imbalance_shooting(config)
        else:
            config = self.generate_config_lines_for_const_shooting(config)

        with codecs.open(config_file_name, 'w', encoding='utf-8') as f:
            f.write(config)
        config_resource.ready()

    def generate_config_lines_for_const_shooting(self, config):
        log.info('Adding config lines for CONST shooting')
        rps = int(self.Parameters.total_rps)
        start_rps = max(int(rps * 0.05), 1)
        rps_schedule = 'const({}, 5m) line({},{},2m) const({}, {}m)' \
            .format(start_rps, start_rps, rps, rps, self.Parameters.test_duration_minutes)
        return config.format(schedule=rps_schedule, autostop_enabled='false',
                             autostop_config='')

    def generate_config_lines_for_imbalance_shooting(self, config):
        log.info('Adding config lines for IMBALANCE shooting')
        rps_schedule = 'const(1, 5m) step({}, {}, {}, {}s)'.format(self.Parameters.start_rps,
                                                                   self.Parameters.target_rps,
                                                                   self.Parameters.rps_increase_step,
                                                                   self.Parameters.rps_step_duration_seconds)
        log.info('Using autostop config: ' + str(self.Parameters.autostop_config))
        autostop_config = '- ' + '\n  - '.join(self.Parameters.autostop_config.split(' '))
        return config.format(schedule=rps_schedule, autostop_enabled='true',
                             autostop_config=autostop_config)

    def read_shooting_config_template(self):
        from library.python import resource
        return resource.find(
            'sandbox/projects/market/checkout/MarketCheckouterLoadGenerateAmmoFullLoad/shooting_config/check.yaml.tmpl')

    def get_endpoints_rps_lines(self):
        is_custom_shooting = bool(self.Parameters.custom_endpoints_rps)
        if is_custom_shooting:
            return is_custom_shooting, self.Parameters.custom_endpoints_rps.splitlines()
        else:
            rps_resource_path = str(sdk2.ResourceData(self.Parameters.endpoints_rps_resource).path)
            log.debug('opening endpoints_rps_resource: ' + rps_resource_path)
            with open(rps_resource_path) as rps_file:
                return is_custom_shooting, rps_file.readlines()

    def create_ammo_metadata(self):
        meta = []
        is_custom_shooting, rps_lines = self.get_endpoints_rps_lines()
        rps_rows = csv.reader(rps_lines, delimiter=',', quotechar='"')
        supported_methods = self.supported_methods.copy()
        covered_rps_part = 0

        for row in rps_rows:
            log.debug('reading row: ' + str(row))
            http_method = row[0]
            method_name = row[1]
            archived = row[2]
            rps_multiplier = float(row[3])

            if (http_method, method_name) not in supported_methods:
                log.warn('Attention! Method not supported: %s %s %s' % (http_method, method_name, archived))
                continue

            covered_rps_part += rps_multiplier
            meta.append({
                'http_method': http_method,
                'method': method_name,
                'archived': archived,
                'rps_multiplier': rps_multiplier,
                'generator': supported_methods.pop((http_method, method_name))
            })

        if not is_custom_shooting and len(supported_methods) > 0:
            raise Exception('Not all supported methods have RPS info: {}'.format(supported_methods))
        if covered_rps_part < 0.9:
            log.warn('Covered RPS part is too low: {}! You need to implement ammo for not covered endpoints'
                     .format(covered_rps_part))
        else:
            log.info('covered RPS part: {}'.format(covered_rps_part))
        meta = self.calculate_relative_rps_ratios(meta)
        return meta

    def calculate_relative_rps_ratios(self, meta):
        rps_sum = sum(map(lambda m: m['rps_multiplier'], meta))
        ratio_sum = 0
        for m in meta:
            m['rps_multiplier'] /= rps_sum
            ratio_sum += m['rps_multiplier']
            log.debug(m['http_method'] + ' ' + m['method'] + ' part is ' + str(m['rps_multiplier']))
        log.debug('ratio_sum={} (expected is 1)'.format(ratio_sum))
        return meta

    def create_options(self):
        from data_base import DataBase

        archived_db_password = sdk2.Vault.data('MARKET', 'checkouter_load_archive_db_password')
        db_password = sdk2.Vault.data('MARKET', 'checkouter_load_db_password')

        db = DataBase(db_password,
                      'market_checkouter_prod',
                      'market_checkouter',
                      'sas-cly1hsv1shkv06r8.db.yandex.net')
        archive_db = DataBase(archived_db_password,
                              'market_checkouter_archive_prod',
                              'market_checkouter_archive',
                              'sas-7muh8t46vqeyvge4.db.yandex.net')
        order_ids = db.fetch_order_ids()
        existing_users = db.fetch_existing_user_ids()
        order_ids_receipt_ids = db.fetch_order_ids_receipt_ids()
        processing_orders = db.fetch_processing_order_ids()
        payment_ids = db.fetch_payment_ids()
        order_ids_with_payment = db.fetch_order_ids_with_payment()
        archived_order_ids = archive_db.fetch_order_ids()
        archived_existing_users = archive_db.fetch_existing_user_ids()
        archived_order_ids_receipt_ids = archive_db.fetch_order_ids_receipt_ids()
        archived_payment_ids = archive_db.fetch_payment_ids()
        archived_order_ids_with_payment = archive_db.fetch_order_ids_with_payment()

        offer, feed_id, offer_id, fee_show, ware_id, user, delivery_id, outlet = self.get_delivery_info(existing_users)

        return {
            'existing_users': existing_users,
            'available_orders': order_ids,
            'processing_orders': processing_orders,
            'order_ids_receipt_ids': order_ids_receipt_ids,
            'payment_ids': payment_ids,
            'order_ids_with_payment': order_ids_with_payment,
            'archived_existing_users': archived_existing_users,
            'archived_available_orders': archived_order_ids,
            'archived_order_ids_receipt_ids': archived_order_ids_receipt_ids,
            'archived_payment_ids': archived_payment_ids,
            'archived_order_ids_with_payment': archived_order_ids_with_payment,
            'latest_orders': xrange(self.Parameters.latest_order_from, self.Parameters.latest_order_to),
            'available_promocodes': self.Parameters.available_promocodes.split(','),

            'loyalty_url': self.Parameters.loyalty_url,

            'coins_promo_id': self.Parameters.coins_promo_id,
            'shop_id': self.Parameters.shop_id,
            'last_event_id_range': xrange(self.Parameters.last_event_id_from, self.Parameters.last_event_id_range),

            'ware_id': ware_id,
            'feed_id': feed_id,
            'offer_id': offer_id,
            'fee_show': fee_show,
            'supplier_id': offer['supplier']['id'],
            'shop_sku': offer['shopSku'],
            'offer_price': offer['prices']['value'],
            'warehouse_id': offer['supplier']['warehouseId'],
            'delivery_id': delivery_id,
            'outlet': outlet,
            'available_track_ids': xrange(0, 1000000),
            'available_track_codes': xrange(1000000, 100000000000),
            'available_checkpoint_ids': xrange(0, 1000000),
        }

    def get_delivery_info(self, existing_users):
        return self.get_delivery_info_with_retries(existing_users, 100)

    def get_delivery_info_with_retries(self, existing_users, tries=100, last_exception=None):
        if tries <= 0:
            raise last_exception
        try:
            offer = get_offer(self.Parameters.report_url, self.Parameters.shop_id, self.Parameters.datafeed_id)
            feed_id = offer['shop']['feed']['id']
            offer_id = offer['shop']['feed']['offerId']
            fee_show = offer['feeShow']
            ware_id = offer['wareId']

            user = random.choice(existing_users)
            delivery_id, outlet = get_delivery_id(self.Parameters.checkouter_url, user,
                                                  self.Parameters.shop_id, feed_id, offer_id, fee_show,
                                                  self.Parameters.delivery_type)

            return offer, feed_id, offer_id, fee_show, ware_id, user, delivery_id, outlet
        except Exception as e:
            tries = tries - 1
            log.warning('tries left: {}'.format(tries))
            return self.get_delivery_info_with_retries(existing_users, tries, e)
