# -*- coding: utf8 -*-

from sandbox import sdk2
from sandbox.projects.market.checkout.helpers.get_offer import get_offer

import collections
import datetime
import logging
import random
import requests

class MarketCheckouterLoadPrepare(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        push_api_url = sdk2.parameters.String('Push api url', default='http://push-api.load.vs.market.yandex.net:39002')
        checkouter_url = sdk2.parameters.String('Checkouter url', default='http://checkouter.load.vs.market.yandex.net:39001')
        stock_storage_url = sdk2.parameters.String('Stock storage url', default='https://bos.tst.vs.market.yandex.net/stocks')
        report_url = sdk2.parameters.String('Report url', default='http://report.tst.vs.market.yandex.net:17051')

        shop_id = sdk2.parameters.Integer('Shop id', default=431782)
        delivery_type = sdk2.parameters.String('Delivery type', default='DELIVERY')
        datafeed_id = sdk2.parameters.Integer('Datafeed id', default=200344277)

    def on_execute(self):
        offer = get_offer(self.Parameters.report_url, self.Parameters.shop_id, self.Parameters.datafeed_id)
        self.feed_id = offer['shop']['feed']['id']
        self.offer_id = offer['shop']['feed']['offerId']
        self.fee_show = offer['feeShow']
        self.supplier_id = offer['supplier']['id']
        self.shop_sku = offer['shopSku']
        self.offer_price = offer['prices']['value']
        self.warehouse_id = offer['supplier']['warehouseId']
        self.available_users = xrange(0, 500000)

        self.setup_meta_data()
        self.setup_push_api()
        self.setup_stocks()
        self.setup_delivery()

    def setup_push_api(self):
        push_api_url = self.Parameters.push_api_url
        logging.info('Trying to post push api settings')

        req = '{0}/shops/{1}/settings'.format(push_api_url, self.Parameters.shop_id)

        try:
            headers = {'Content-Type': 'application/xml'}
            payload = '''<settings partner-interface="true"/>'''

            logging.info('request: %s, payload: %s', req, payload)
            push_api_answ = requests.post(req, data=payload, headers=headers)
            logging.info('Code: %s', push_api_answ.status_code)
            if push_api_answ.status_code != 200:
                raise Exception()
            return push_api_answ
        except Exception as exc:
            logging.exception(
                'Exception trying to setup push_api: %s. Url %s', exc, req)
            raise exc

    def setup_meta_data(self):
        try:
            logging.info('Trying to post shop metadata')
            for shopId in [self.Parameters.shop_id, self.supplier_id]:
                req = '{0}/shops/{1}'.format(self.Parameters.checkouter_url, shopId)
                headers = {'Content-Type': 'application/json'}
                payload = '''
                    {
                        "campaignId": 21421814,
                        "clientId": 56164479,
                        "sandboxClass": "YANDEX",
                        "prodClass": "YANDEX",
                        "prepayType": "YANDEX_MARKET",
                        "inn": "6449013711",
                        "phoneNumber": "+23456876543567"
                    }
                '''

                logging.info('request: %s, payload: %s', req, payload)
                meta_data_answ = requests.put(
                    req, data=payload, headers=headers)
                logging.info('Code: %d', meta_data_answ.status_code)
                if meta_data_answ.status_code != 200:
                    raise Exception()
        except Exception as exc:
            logging.exception(
                'Exception trying to setup metadata: %s. Url %s', exc, req)

    def setup_stocks(self):
        json = {
            'itemStocks': [
                {
                    'unitId': {
                        'id': self.shop_sku,
                        'vendorId': self.supplier_id,
                        'article': self.shop_sku
                    },
                    'warehouseId': {
                        'yandexId': self.warehouse_id,
                        'fulfillmentId': 'dj389hd'
                    },
                    'stocks': [
                        {'type': 1, 'count': 100000, "updated": str(datetime.datetime.now().strftime("%Y-%m-%dT%H:%M:%S"))}
                    ]
                }
            ],
            "source": {
                "warehouseId": self.warehouse_id
            }
        }

        logging.info('posting stocks. Req: %s, json: %s', self.Parameters.stock_storage_url, json)

        stocks_answ = requests.post(self.Parameters.stock_storage_url, json=json)
        if stocks_answ.status_code != 200:
            raise Exception('cannot push stocks')

    def setup_delivery(self):
        req = '{0}/cart?uid={1}&rgb=BLUE'.format(self.Parameters.checkouter_url, random.choice(self.available_users))

        headers = {'Content-Type': 'application/json',
                   'X-Hit-Rate-Group': 'UNLIMIT'}

        payload = u'''{
            "buyerRegionId": %d,
            "buyerCurrency": "RUR",
            "carts": [
                {
                    "shopId": %s,
                    "items": [
                        {
                            "feedId": %s,
                            "offerId": "%s",
                            "showInfo": "%s",
                            "count": 1
                        }
                    ],
                    "delivery" : {
                        "regionId" : %d,
                        "address" : {
                            "country" : "Россия",
                            "postcode" : "119313",
                            "city" : "Санкт-Петербург",
                            "subway" : "Проспект Вернадского",
                            "street" : "Ленинский проспект",
                            "house" : "90",
                            "floor" : "6"
                        }
                    }
                }
            ]
        }''' % (213, self.Parameters.shop_id, self.feed_id, self.offer_id, self.fee_show, 213)

        logging.info("Getting delivery id: %s", req)

        get_delivery_id_answ = requests.post(
            req, data=payload.encode('utf-8'), headers=headers)
        if get_delivery_id_answ.status_code != 200:
            self.delivery_id = None

            logging.info(get_delivery_id_answ.headers)

            raise Exception()
        delivery_id = None
        for delivery in get_delivery_id_answ.json()['carts'][0]['deliveryOptions']:
            delivery_type = delivery['type']
            if delivery_type != self.Parameters.delivery_type:
                continue
            delivery_id = delivery['id']
            if delivery_type == 'PICKUP':
                outlet = '"outletId": ' + str(delivery['outlets'][0]['id'])
            elif delivery['type'] == 'DELIVERY':
                outlet = u'''
                    "address": {
                        "country": "Русь",
                        "postcode": "131488",
                        "city": "Питер",
                        "subway": "Петровско-Разумовская",
                        "street": "Победы",
                        "house": "13",
                        "block": "666",
                        "entrance": "404",
                        "entryphone": "007",
                        "floor": "8",
                        "apartment": "303",
                        "recipient": "000",
                        "phone": "02"
                    }
                '''
            break
        if not delivery_id:
            self.delivery_id = None
            logging.error('no delivery id in get delivery id answer')
            raise Exception()

        self.delivery_id = delivery_id
        self.outlet = outlet
