import logging
import os
import tarfile

import sandbox.common.types.client as ctc

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment

class MarketCheckouterRunMigrations(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = [SandboxJavaJdkEnvironment('11.0.2')]
        client_tags = ctc.Tag.Group.LINUX


    class Parameters(sdk2.Task.Parameters):
        checkouter_db_app_resource = sdk2.parameters.Resource('checkouter-db app resource')
        main_datasources = sdk2.parameters.Resource('main environment datasources')
        additional_datasources = sdk2.parameters.Resource('additional checkouter datasources')
        additional_carter_datasources = sdk2.parameters.Resource('additional carter datasources')
        additional_notifier_datasources = sdk2.parameters.Resource('additional notifier datasources')

        environment = sdk2.parameters.String('environment')

        # Remove after migration to yav
        vault_owner = sdk2.parameters.String('Vault passwords token owner', default='MARKET')
        secrets = sdk2.parameters.List('list of secrets', sdk2.parameters.String, default=[])
        database_type = sdk2.parameters.String('database_type', default='PRODUCTION')



    def on_execute(self):
        self.workdir = self.path('checkouter-db').as_posix()

        self.prepare_resource(self.Parameters.checkouter_db_app_resource)
        self.prepare_resource(self.Parameters.main_datasources)
        if hasattr(self.Parameters, 'additional_datasources') and self.Parameters.additional_datasources is not None:
            self.prepare_resource(self.Parameters.additional_datasources)

        if hasattr(self.Parameters, 'additional_carter_datasources') and self.Parameters.additional_carter_datasources is not None:
            self.prepare_resource(self.Parameters.additional_carter_datasources)

        if hasattr(self.Parameters, 'additional_notifier_datasources') and self.Parameters.additional_notifier_datasources is not None:
            self.prepare_resource(self.Parameters.additional_notifier_datasources)

        if hasattr(self.Parameters, 'secrets') and self.Parameters.secrets is not None and len(self.Parameters.secrets) > 0:
            # secrets reading should be in method on_*, except some cases
            secret_path = os.path.join(self.path('checkouter-db').as_posix(), 'properties.d', '75_checkouter_secrets.properties')

            with open(secret_path, 'w') as f:
                for secret in self.Parameters.secrets:
                    if secret.startswith("sec-"):
                        # secrets from yav
                        yav_secret = sdk2.yav.Secret(secret)
                        secret_data = yav_secret.data()
                        for key in secret_data.keys():
                            f.write('{}={}\n'.format(key, secret_data[key]))
                    else:
                        # secrets from sandbox (deprecated)
                        value = sdk2.Vault.data(self.Parameters.vault_owner, secret)
                        f.write('{}={}\n'.format(secret, value))

        script_name = 'market-checkouter-db-start.sh'
        if hasattr(self.Parameters, 'database_type'):
            database_type = self.Parameters.database_type
            script_name = {
                'PRODUCTION': 'market-checkouter-db-start.sh',
                'PRODUCTION_TMS': 'market-checkouter-db-tms-start.sh',
                'ARCHIVE': 'market-checkouter-db-start-archive.sh',
                'MSTAT': 'market-checkouter-db-start-mstat.sh',
                'CARTER': 'market-checkouter-db-start-carter.sh',
                'NOTIFIER': 'market-notifier-db-start-notifier.sh',
                'NOTIFIER_TMS': 'market-notifier-db--tms-start-notifier.sh'
            }.get(database_type)


        args = ' '.join([
            os.path.join(self.workdir, 'bin', script_name),
            '--environment={}'.format(self.Parameters.environment),
            '--java-home={}'.format(os.environ['JAVA_HOME'])
        ])

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("market-checkouter-db")) as pl:
            return_code = sp.Popen(args, shell=True, stdout=pl.stdout, stderr=sp.STDOUT, cwd=self.workdir).wait()
            if return_code != 0:
                raise Exception('Return code is not zero: ' + str(return_code))


    def prepare_resource(self, resource):
        resource = sdk2.ResourceData(resource)
        resource_path = resource.path.as_posix()

        logging.info('resource path: %s', resource_path)

        self.extract_with_strip(resource_path)

    def extract_with_strip(self, tar_path, strip=1):
        tar = tarfile.open(tar_path)
        for member in tar:
            split = member.name.split(os.path.sep)[strip:]
            if len(split) == 0:
                continue
            member.name = os.path.join(*split)
            tar.extract(member, path=self.workdir)
