import logging
import os
import zipfile

import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.constants import constants as sdk_constants
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.sdk2.helpers import subprocess as sp

base_path = 'sandbox/projects/market/checkout/MarketCheckouterRunSonar/'


def read_resource(file_name):
    import library.python.resource as resource
    return resource.find(base_path + file_name)


class MarketCheckouterRunSonar(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = [SandboxJavaJdkEnvironment('11.0.2')]
        client_tags = ctc.Tag.Group.LINUX

    class Parameters(sdk2.Task.Parameters):
        checkout_arcadia_from_url = sdk2.parameters.String('checkout_arcadia_from_url', default='')
        pull_request_key = sdk2.parameters.String('pull_request_key', default='')
        pull_request_branch = sdk2.parameters.String('pull_request_branch', default='')
        pull_request_base = sdk2.parameters.String('pull_request_base', default='')
        revision = sdk2.parameters.String('revision', default='')
        build_target = sdk2.parameters.String('build_target', default='')
        project_key = sdk2.parameters.String('project_key', default='')
        sources = sdk2.parameters.String('sources', default='')

    def on_execute(self):
        current_dir = '.'

        # mount arcadia
        output_directory = os.path.join(current_dir, 'output')
        build_target = os.path.join(current_dir, self.Parameters.build_target)

        with arcadiasdk.mount_arc_path(self.Parameters.checkout_arcadia_from_url) as arcadia_path:
            arcadiasdk.do_build(
                build_system=sdk_constants.YMAKE_BUILD_SYSTEM,
                source_root=arcadia_path,
                targets=[build_target],
                results_dir=output_directory,
                clear_build=False)

            # running sonar-scanner
        sonar_scanner_zip = read_resource('sonar-scanner-cli-4.6.2.2472-linux.zip')
        scanner_zip = 'sonar-scanner.zip'
        with open(scanner_zip, 'w') as f:
            f.write(sonar_scanner_zip)

        with zipfile.ZipFile(scanner_zip, 'r') as zip_ref:
            zip_ref.extractall(current_dir)

        sonar_scanner = os.path.join(current_dir, 'bin', 'sonar-scanner')

        pr_key = self.Parameters.pull_request_key
        pr_branch = self.Parameters.pull_request_branch
        pr_base = self.Parameters.pull_request_base
        branch_params = '-Dsonar.pullrequest.key=%s -Dsonar.pullrequest.branch=%s -Dsonar.pullrequest.base=%s' % (
        pr_key, pr_branch, pr_base)
        args = ' '.join([
            sonar_scanner,
            branch_params if self.Parameters.pull_request_branch else ''
        ])

        sonar_properties = os.path.join(current_dir, 'sonar-project.properties')

        project_key = self.Parameters.project_key
        sources = self.Parameters.sources
        if '%ARCADIA_PATH%' in sources:
            sources = sources.replace('%ARCADIA_PATH%', arcadia_path)

        with open(sonar_properties, 'w') as f:
            f.write('sonar.projectKey=' + project_key)
            f.write('sonar.projectVersion=1.0')
            f.write('sonar.sourceEncoding=UTF-8')
            f.write('sonar.sources=' + sources)
            f.write('sonar.java.binaries=' + output_directory)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("sonar-scanner")) as pl:
            return_code = sp.Popen(args, shell=True, stdout=pl.stdout, stderr=sp.STDOUT, cwd=current_dir).wait()
            if return_code != 0:
                raise Exception('Return code is not zero: ' + str(return_code))
