# coding=UTF8
import requests
import logging
import json
import datetime
from sandbox.projects.release_machine.helpers.staff_helper import StaffApi


class Notificator:

    def __init__(self,
                 yav_tokens,
                 calendar_id,
                 notification_chat_id,
                 enable_pin,
                 enable_unpin,
                 hotline_chat_url,
                 support_chat_url):
        self.startrek_oauth_token = yav_tokens.data()['startrek_oauth_token']
        self.telegram_bot_token = yav_tokens.data()['telegram_bot_token']
        self.abc_token = yav_tokens.data()['abc_token']
        self.telegram_api_url = 'https://api.telegram.org'
        self._staff = StaffApi(self.startrek_oauth_token)
        self.staff_url = "https://staff.yandex-team.ru/{}"
        self.logger = logging.getLogger(__name__)
        self.abc_api_url = 'https://abc-back.yandex-team.ru/api/v4/duty/shifts/?service__slug=checkouter&schedule={}&date_from={}&date_to={}&with_watcher=1'
        self.questions_form_url = 'https://st.yandex-team.ru/createTicket?queue=MARKETCHECKOUT&_form=88520'
        self.duty_schedule_url = 'https://abc.yandex-team.ru/services/checkouter/duty2/'
        self.chat_list_url = 'https://wiki.yandex-team.ru/Market/Development/incident-routing/#checkoutercarternotifier'
        self.support_dashboard_url = 'https://st.yandex-team.ru/dashboard/29128'
        self.create_support_ticket_url = 'https://st.yandex-team.ru/createTicket?queue=MARKETCHECKOUT'

        self.notification_chat_id = notification_chat_id
        self.calendar_id = calendar_id
        self.enable_pin = enable_pin
        self.enable_unpin = enable_unpin
        self.hotline_chat_url = hotline_chat_url
        self.support_chat_url = support_chat_url

    def do_notification(self):
        today = datetime.datetime.today().replace(hour=0, minute=0, second=0, microsecond=0)
        sunday = self.next_weekday(today, 6)
        duties = self.get_current_duties(today, sunday)
        if self.enable_unpin:
            self.unpin_chat_message()
        result = self.send_telegram_message(self.get_telegram_message(duties, today, sunday))
        if self.enable_pin:
            self.pin_chat_message(result['result']['message_id'])

    def get_current_duties(self, today, sunday):
        duties = \
            requests.get(
                self.abc_api_url.format(self.calendar_id, today.strftime("%Y-%m-%d"), sunday.strftime("%Y-%m-%d")),
                headers={'Authorization': 'OAuth {}'.format(self.abc_token)}).json()['results']
        self.logger.info("Duties found: {}".format(duties))
        return duties

    def send_telegram_message(self, message):
        data = json.dumps({
            "chat_id": self.notification_chat_id,
            "text": message,
            "parse_mode": "HTML",
            "method": "sendMessage"
        })
        result = requests.post(
            '{}/bot{}/sendMessage'.format(self.telegram_api_url, self.telegram_bot_token),
            headers={'Content-Type': 'application/json'},
            data=data
        ).json()
        self.logger.info("Sending notification result {}".format(result))
        return result

    def get_telegram_message(self, duties, today, sunday):
        html_link_pattern = "<a href=\"{}\">{}</a>"
        message = '<b>Дежурства на неделе с {} по {}: </b>\n'.format(today.strftime("%Y-%m-%d"), sunday.strftime("%Y-%m-%d"))
        for duty in duties:
            message = message + "С {} по {} дежурит в графике {}: @{} " \
                .format(duty['start'], duty['end'], duty['schedule']['name'],
                        self.get_telegram_contact(duty['person']['login'])) + \
                      html_link_pattern.format(self.staff_url.format(duty['person']['login']), "Стафф") + '\n'
        if not duties:
            message = message + "На этой неделе нет поддержки. Приходите на следующей неделе, или заведите " \
                                + html_link_pattern.format(self.create_support_ticket_url, "тикет") + "\n"
            message = message + "В случае инцидента пишите в hotline-чат\n"
        message = message + "\n\n"
        message = message + html_link_pattern.format(self.support_chat_url, "Ссылка на этот чат") + " \n"
        message = message + html_link_pattern.format(self.hotline_chat_url, "Hotline-чат") + " в случае инцидента\n"
        message = message + html_link_pattern.format(self.questions_form_url, "Форма для вопросов") + "\n"
        message = message + html_link_pattern.format(self.support_dashboard_url, "Доска поддержки") + "\n"
        message = message + html_link_pattern.format(self.duty_schedule_url, "График дежурств") + "\n"
        message = message + html_link_pattern.format(self.chat_list_url, "Список чатов") + "\n"
        self.logger.info("Message to send:  {}".format(message))
        return message

    def next_weekday(self, d, weekday):
        days_ahead = weekday - d.weekday()
        if days_ahead <= 0:  # Target day already happened this week
            days_ahead += 7
        return d + datetime.timedelta(days_ahead)

    def get_telegram_contact(self, login):
        contacts = self._staff.get_person_profile_property(login, 'telegram_accounts')['telegram_accounts']
        for contact in contacts:
            if 'bot' not in contact['value']:
                return contact['value']

    def pin_chat_message(self, message_id):
        data = json.dumps({
            "chat_id": self.notification_chat_id,
            "message_id": message_id,
            "disable_notification": "true"
        })
        result = requests.post(
            '{}/bot{}/pinChatMessage'.format(self.telegram_api_url, self.telegram_bot_token),
            data=data,
            headers={'Content-Type': 'application/json'}
        ).json()
        self.logger.info("Result of pining {}".format(result))

    # unpins last message
    def unpin_chat_message(self):
        result = requests.post(
            '{}/bot{}/unpinChatMessage'.format(self.telegram_api_url, self.telegram_bot_token),
            data=json.dumps({
                "chat_id": self.notification_chat_id,
            }),
            headers={'Content-Type': 'application/json'}
        ).json()
        self.logger.info("Result of unpining {}".format(result))
