import sys
import os

MIN_MARKET_DAU_CAPACITY_SENSOR = 'minMarketDauCapacity'

MIN_MARKET_ORDERS_CAPACITY_SENSOR = 'minMarketOrdersCapacity'

arc_root_path = os.getenv('ARC_ROOT')
if arc_root_path is not None:
    sys.path.extend([arc_root_path])
    sys.path.extend([arc_root_path + 'market/sre/tools/capacity'])
import logging as log

import pandas as pd
from datetime import datetime, timedelta

from sandbox.projects.market.checkout.MarketLoadTestResultProcessor.solomon_client import SolomonClient

CLICKHOUSE_URL = 'clickhouse-public.market.yandex.net'

SQL_GET_ORDERS_STATS = """
        select
            newStatus as status,
            date,
            count(*) as orders_count
        from
            market.checkouter_order_history
        where
            date between '{date_start}' and '{date_end}'
            and context='MARKET'
            and uid <> 2308324861409815965
            and not (uid between 2190550858753437195 and 2190550859753437194)
            and orderId > 10000
            and eventType='ORDER_STATUS_UPDATED'
            and newStatus in ('PROCESSING', 'RESERVED')
        group by
            status,
            date
        order by
            status desc,
            date desc
        """

DAILY_ORDERS_COUNT_SENSOR = 'dailyOrdersCount'

CHECKOUTER_ORDERS_FORECAST_SENSOR = 'ordersForecast'

CHECKOUTER_RPS_FORECAST_SENSOR = 'rpsForecast'

CARTER_RPS_FORECAST_SENSOR = 'rpsForecastCarter'

LOYALTY_RPS_FORECAST_SENSOR = 'rpsForecastLoyalty'

CARTER_DAU_CAPACITY_SENSOR = 'maxCarterDau_2'
CARTER_DAU_CAPACITY_DC_1_SENSOR = 'maxCarterDauDcMinus1_2'

CHECKOUTER_ORDERS_CAPACITY_SENSOR = 'maxCheckouterOrders_2'
CHECKOUTER_ORDERS_CAPACITY_DC_1_SENSOR = 'maxCheckouterOrdersDcMinus1_2'

CHECKOUTER_RPS_CAPACITY_SENSOR = 'maxRps_3'
CHECKOUTER_RPS_CAPACITY_DC_1_SENSOR = 'maxRpsDcMinus1_3'

LOYALTY_RPS_CAPACITY_SENSOR = 'loyaltyMaxRps_5'
LOYALTY_RPS_CAPACITY_DC_1_SENSOR = 'loyaltyMaxRpsDcMinus1_2'

INSTANCE_RPS_CAPACITY_SENSOR = 'maxInstanceRps'

CHECKOUTER_RESERVED_ORDERS_CAPACITY_DC_1_SENSOR = 'reservedOrdersCapacityDcMinus1'
CHECKOUTER_RESERVED_ORDERS_CAPACITY_SENSOR = 'reservedOrdersCapacity'

CHECKOUTER_RESERVED_FPS_CAPACITY_SENSOR = 'reservedFpsCapacity'
CHECKOUTER_RESERVED_FPS_CAPACITY_DC_1_SENSOR = 'reservedFpsCapacityDcMinus1'
CHECKOUTER_RESERVED_MIN_CAPACITY_SENSOR = 'reservedMinCapacity'
CHECKOUTER_RESERVED_MIN_CAPACITY_DC_1_SENSOR = 'reservedMinCapacityDcMinus1'

CARTER_RESERVED_DAU_CAPACITY_DC_1_SENSOR = 'carter_ReservedDauCapacityDcMinus1'
CARTER_RESERVED_DAU_CAPACITY_SENSOR = 'carter_reservedDauCapacity'

CARTER_RESERVED_FPS_CAPACITY_SENSOR = 'carter_reservedFpsCapacity'
CARTER_RESERVED_FPS_CAPACITY_DC_1_SENSOR = 'carter_reservedFpsCapacityDcMinus1'
CARTER_RESERVED_MIN_CAPACITY_SENSOR = 'carter_reservedMinCapacity'
CARTER_RESERVED_MIN_CAPACITY_DC_1_SENSOR = 'carter_reservedMinCapacityDcMinus1'

CARTER_DAU_FORECAST_SENSOR = 'dauForecast'

CARTER_MAX_PROD_RPS_SENSOR = 'carterMaxProdRps'

CHECKOUTER_MAX_PROD_RPS_SENSOR = 'maxProdRps'

CARTER_RPS_CAPACITY_SENSOR = 'carterMaxRps_5'
CARTER_RPS_CAPACITY_DC_1_SENSOR = 'carterMaxRpsDcMinus1_5'

ORDERS_FORECAST = {
    '2021-01-01': 83635,
    '2021-02-01': 144923,
    '2021-03-01': 143242,
    '2021-04-01': 147676,
    '2021-05-01': 149604,
    '2021-06-01': 151573,
    '2021-07-01': 150452,
    '2021-08-01': 168065,
    '2021-09-01': 192116,
    '2021-10-01': 195680,
    '2021-11-01': 352067,
    '2021-12-01': 372571,
    '2022-01-01': 192366,
    '2022-02-01': 260278,
    '2022-03-01': 260562,
    '2022-04-01': 242846,
    '2022-05-01': 237646,
    '2022-06-01': 248974,
    '2022-07-01': 257644,
    '2022-08-01': 270282,
    '2022-09-01': 325696,
    '2022-10-01': 329700,
    '2022-11-01': 609767,
    '2022-12-01': 645454,
}

DAU_FORECAST = {
    '2021-01-01': 6590000,
    '2021-02-01': 6773000,
    '2021-03-01': 6758000,
    '2021-04-01': 6525000,
    '2021-05-01': 6614000,
    '2021-06-01': 6681000,
    '2021-07-01': 6630000,
    '2021-08-01': 6422000,
    '2021-09-01': 6763000,
    '2021-10-01': 6925000,
    '2021-11-01': 7333000,
    '2021-12-01': 7286000,
    '2022-01-01': 7596000,
    '2022-02-01': 7847000,
    '2022-03-01': 7815000,
    '2022-04-01': 7516000,
    '2022-05-01': 7595000,
    '2022-06-01': 7692000,
}

CHECKOUTER_RPS_ALL_KEY = 'sumSeries(five_min.checkouter.STABLE.{POST,GET,DELETE,PATCH,PUT}.rps)'
CARTER_RPS_ALL_KEY = 'five_min.market-carter.STABLE.rps'


# CHECKOUTER_NEW_ORDER_COUNT_KEY = 'timeShift(one_day.checkouter.monitorings.BIZ.rgb.BLUE.order.new.TOTAL, "-1d")'

class ShootingsProcessor:
    def __init__(self, solomon_token, yql_token):
        self.yql_token = yql_token
        self.solomon_token = solomon_token
        self.solomon = SolomonClient(oauth_token=self.solomon_token, project_id='market-checkout',
                                     cluster_name='stable', solomon_api_url='http://solomon.yandex.net')

    def process(self):
        # Forecasts - TODO get data from KPI service when its ready
        # self.make_checkouter_forecast()
        # self.make_carter_forecast()

        self.process_loyalty()

        self.process_checkouter()

        self.process_carter()

        self.calc_min_capacity()

    def process_checkouter(self):
        log.info('processing checkouter')
        # Calculate max component RPS
        from lib.Service import RTC
        self.init_checkouter_df()
        checkouter_services = RTC.get_services(_filter=['checkouter', '~planeshift', '~_man'])
        checkouter_dc_names = ['SAS', 'IVA', 'VLA']
        current_checkouter_instance_rps_capacity = self.calc_rps_capacity(
            solomon_service_name='market-checkouter',
            rps_capacity_sensor=CHECKOUTER_RPS_CAPACITY_SENSOR,
            rps_capacity_dc_1_sensor=CHECKOUTER_RPS_CAPACITY_DC_1_SENSOR,
            services=checkouter_services,
            dc_names=checkouter_dc_names)
        self.calc_hardware(services=checkouter_services,
                           dc_names=checkouter_dc_names,
                           rps_forecast_sensor=CHECKOUTER_RPS_FORECAST_SENSOR,
                           sensor_prefix='',
                           current_instance_rps_capacity=current_checkouter_instance_rps_capacity)
        # RPS capacity TO KPI
        self.calc_checkouter_orders_capacity()
        # Calculate actual max prod RPS
        self.calc_actual_prod_rps(CHECKOUTER_RPS_ALL_KEY, CHECKOUTER_MAX_PROD_RPS_SENSOR)
        self.calc_daily_orders()
        # Calculate capacity reservation
        orders_reserved_by_ts = self.calc_capacity_reservation(CHECKOUTER_ORDERS_CAPACITY_SENSOR,
                                                               CHECKOUTER_ORDERS_FORECAST_SENSOR,
                                                               CHECKOUTER_RESERVED_ORDERS_CAPACITY_SENSOR)
        fps_reserved_by_ts = self.calc_capacity_reservation(CHECKOUTER_RPS_CAPACITY_SENSOR,
                                                            CHECKOUTER_RPS_FORECAST_SENSOR,
                                                            CHECKOUTER_RESERVED_FPS_CAPACITY_SENSOR)
        orders_dc_1_reserved_by_ts = self.calc_capacity_reservation(CHECKOUTER_ORDERS_CAPACITY_DC_1_SENSOR,
                                                                    CHECKOUTER_ORDERS_FORECAST_SENSOR,
                                                                    CHECKOUTER_RESERVED_ORDERS_CAPACITY_DC_1_SENSOR)
        fps_dc_1_reserved_by_ts = self.calc_capacity_reservation(CHECKOUTER_RPS_CAPACITY_DC_1_SENSOR,
                                                                 CHECKOUTER_RPS_FORECAST_SENSOR,
                                                                 CHECKOUTER_RESERVED_FPS_CAPACITY_DC_1_SENSOR)
        self.calc_min_reservation(orders_reserved_by_ts, fps_reserved_by_ts,
                                  CHECKOUTER_RESERVED_MIN_CAPACITY_SENSOR)
        self.calc_min_reservation(orders_dc_1_reserved_by_ts, fps_dc_1_reserved_by_ts,
                                  CHECKOUTER_RESERVED_MIN_CAPACITY_DC_1_SENSOR)

    def process_carter(self):
        log.info('processing carter')
        from lib.Service import RTC
        self.init_carter_df()
        carter_services = RTC.get_services(_filter='carter')
        carter_dc_names = ['SAS', 'IVA', 'VLA']
        current_carter_instance_rps_capacity = self.calc_rps_capacity(
            solomon_service_name='market-carter',
            rps_capacity_sensor=CARTER_RPS_CAPACITY_SENSOR,
            rps_capacity_dc_1_sensor=CARTER_RPS_CAPACITY_DC_1_SENSOR,
            services=carter_services,
            dc_names=carter_dc_names)
        self.calc_hardware(services=carter_services,
                           dc_names=carter_dc_names,
                           rps_forecast_sensor=CARTER_RPS_FORECAST_SENSOR,
                           sensor_prefix='carter_',
                           current_instance_rps_capacity=current_carter_instance_rps_capacity)
        self.calc_carter_dau_capacity()
        self.calc_actual_prod_rps(CARTER_RPS_ALL_KEY, CARTER_MAX_PROD_RPS_SENSOR)
        orders_dc_1_reserved_by_ts = self.calc_capacity_reservation(CARTER_DAU_CAPACITY_DC_1_SENSOR,
                                                                    CARTER_DAU_FORECAST_SENSOR,
                                                                    CARTER_RESERVED_DAU_CAPACITY_DC_1_SENSOR)
        orders_reserved_by_ts = self.calc_capacity_reservation(CARTER_DAU_CAPACITY_SENSOR,
                                                               CARTER_DAU_FORECAST_SENSOR,
                                                               CARTER_RESERVED_DAU_CAPACITY_SENSOR)
        fps_reserved_by_ts = self.calc_capacity_reservation(CARTER_RPS_CAPACITY_SENSOR,
                                                            CARTER_RPS_FORECAST_SENSOR,
                                                            CARTER_RESERVED_FPS_CAPACITY_SENSOR)
        fps_dc_1_reserved_by_ts = self.calc_capacity_reservation(CARTER_RPS_CAPACITY_DC_1_SENSOR,
                                                                 CARTER_RPS_FORECAST_SENSOR,
                                                                 CARTER_RESERVED_FPS_CAPACITY_DC_1_SENSOR)
        self.calc_min_reservation(orders_reserved_by_ts, fps_reserved_by_ts,
                                  CARTER_RESERVED_MIN_CAPACITY_SENSOR)
        self.calc_min_reservation(orders_dc_1_reserved_by_ts, fps_dc_1_reserved_by_ts,
                                  CARTER_RESERVED_MIN_CAPACITY_DC_1_SENSOR)

    def process_loyalty(self):
        log.info('processing loyalty')
        from lib.Service import RTC
        # self.init_loyalty_df() #todo
        loyalty_services = RTC.get_services(_filter='market_loyalty')
        loyalty_dc_names = ['SAS', 'IVA', 'VLA']
        current_loyalty_instance_rps_capacity = self.calc_rps_capacity(
            solomon_service_name='market-loyalty',
            rps_capacity_sensor=LOYALTY_RPS_CAPACITY_SENSOR,
            rps_capacity_dc_1_sensor=LOYALTY_RPS_CAPACITY_DC_1_SENSOR,
            services=loyalty_services,
            dc_names=loyalty_dc_names)
        self.calc_hardware(services=loyalty_services,
                           dc_names=loyalty_dc_names,
                           rps_forecast_sensor=LOYALTY_RPS_FORECAST_SENSOR,
                           sensor_prefix='loyalty_',
                           current_instance_rps_capacity=current_loyalty_instance_rps_capacity)

    def calc_actual_prod_rps(self, graphite_key, sensor_name):
        log.info('calculating actual PROD RPS')
        date_start = (datetime.now() - timedelta(days=3)).strftime('%Y-%m-%d')
        date_end = (datetime.now() - timedelta(days=1)).strftime('%Y-%m-%d')
        from lib.Graphite import Graphite
        max_rps_df = \
            Graphite.get(target=graphite_key, date_from=date_start, date_to=date_end)[['ts', 'value']] \
                .groupby(pd.Grouper(key='ts', freq='1d')) \
                .quantile(.95)[['value']] \
                .rename(columns={'value': 'rps'})
        for ts, row in max_rps_df.iterrows():
            self.solomon.push_sensor('market-checkouter', sensor_name, row['rps'], datetime.timestamp(ts))

    def calc_daily_orders(self):
        log.info('calculating daily orders')
        date_start = (datetime.now() - timedelta(days=2))
        date_end = (datetime.now() - timedelta(days=1))

        orders_history_df = self.get_orders_history_df(date_start, date_end)

        for ts, row in orders_history_df.iterrows():
            self.solomon.push_sensor('market-checkouter', DAILY_ORDERS_COUNT_SENSOR, int(row['orders_count']),
                                     datetime.timestamp(ts))

    def get_orders_history_df(self, date_start, date_end):
        log.info('getting orders history df')
        from lib.Clickhouse import Clickhouse
        CH = Clickhouse(host=CLICKHOUSE_URL)
        if not isinstance(date_start, str):
            date_start = date_start.strftime('%Y-%m-%d')
        if not isinstance(date_end, str):
            date_end = date_end.strftime('%Y-%m-%d')
        sql = SQL_GET_ORDERS_STATS.format(date_start=date_start,
                                          date_end=date_end)
        orders_history_df = CH.query(query=sql, as_df=True)
        orders_history_df = orders_history_df[orders_history_df['status'] == 'RESERVED']
        orders_history_df['date'] = pd.to_datetime(orders_history_df['date'])
        orders_history_df = orders_history_df[['date', 'orders_count']].set_index('date')
        log.debug('orders history df: {}'.format(orders_history_df))
        return orders_history_df

    def calc_capacity_reservation(self, capacity_sensor, forecast_sensor, reserved_capacity_sensor):
        log.info('calculating capacity reservation for {}'.format(reserved_capacity_sensor))
        cap_date_from = datetime.now() - timedelta(days=30)
        cap_timestamps, cap_values = self.solomon.get_values(date_from=cap_date_from,
                                                             service_name='market-checkouter',
                                                             sensor_name=capacity_sensor)

        if len(cap_timestamps) == 0:
            raise Exception('Cannot calculate reservation: no data in "{}"'.format(capacity_sensor))

        reserved_dict = dict()
        forecast_ts_list, forecast_list = self.get_forecast(forecast_sensor)
        today = self.get_today_midnight_timestamp()
        capacity = cap_values[-1]
        today_reserved_percent = (capacity - forecast_list[0]) / forecast_list[0]
        self.solomon.push_sensor(service_name='market-checkouter',
                                 sensor_name=reserved_capacity_sensor,
                                 value=today_reserved_percent, ts=today)
        reserved_dict[today] = today_reserved_percent
        for i in range(len(forecast_ts_list)):
            forecast_reserved_percent = (capacity - forecast_list[i]) / forecast_list[i]

            self.solomon.push_sensor(service_name='market-checkouter',
                                     sensor_name=reserved_capacity_sensor,
                                     value=forecast_reserved_percent, ts=forecast_ts_list[i])
            reserved_dict[forecast_ts_list[i]] = forecast_reserved_percent
        return reserved_dict

    def calc_min_reservation(self, reserved_dict_1, reserved_dict_2, sensor):
        log.info('calcualting minimal reservation % for {}'.format(sensor))
        for ts in reserved_dict_1:
            val1 = reserved_dict_1[ts]
            val2 = reserved_dict_2.get(ts, val1)
            self.solomon.push_sensor(service_name='market-checkouter',
                                     sensor_name=sensor,
                                     value=min(val1, val2), ts=ts)

    def get_today_midnight_timestamp(self):
        return self.timestamp_to_midnight(int(datetime.now().timestamp()))

    def get_forecast(self, forecast_sensor):
        log.info('getting RPS forecast ({})'.format(forecast_sensor))
        forecast_date_from = datetime.now() + timedelta(days=1)
        forecast_date_to = datetime.now() + timedelta(days=445)
        forecast_timestamps, forecast_values = self.solomon.get_values(date_from=forecast_date_from,
                                                                       date_to=forecast_date_to,
                                                                       service_name='market-checkouter',
                                                                       sensor_name=forecast_sensor)

        if len(forecast_timestamps) == 0:
            raise Exception('Forecast data "{}" not found in Solomon'.format(forecast_sensor))

        return forecast_timestamps, forecast_values

    def calc_rps_capacity(self, solomon_service_name, dc_names, rps_capacity_sensor, rps_capacity_dc_1_sensor,
                          services):
        log.info('calculating RPS capacity for {} {}'.format(solomon_service_name, rps_capacity_sensor))
        date_from = datetime.now() - timedelta(days=30)
        instance_capacity_ts_list, instance_capacity_list = self.solomon.get_values(
            date_from=date_from,
            service_name=solomon_service_name,
            sensor_name=INSTANCE_RPS_CAPACITY_SENSOR,
            smooth=True)

        if len(instance_capacity_ts_list) == 0:
            raise Exception('Cannot calculate RPS capacity: no data in "{}"'.format(INSTANCE_RPS_CAPACITY_SENSOR))

        from lib.Service import Service, RTC
        from functools import partial
        get_rtc_config = partial(RTC.get_services_instances_count, services=services)

        for i in range(len(instance_capacity_ts_list)):
            rps_capacity_ts = self.timestamp_to_midnight(instance_capacity_ts_list[i])
            maxInstanceRps = instance_capacity_list[i]
            log.debug('processing value {} ts {}'.format(maxInstanceRps, rps_capacity_ts))
            service = Service(name="service", current_rps=1, dead_instances_per_dc_max=2)
            for dc_name in dc_names:
                service.add_instances(dc=dc_name, instances_count=get_rtc_config(dc=dc_name),
                                      instance_rps_limit=maxInstanceRps)
            maxRps = service.calc_rps_capacity(dc_minus_1=False, consider_reload=True, peak_coef=1)
            maxRpsDcMinus1 = service.calc_rps_capacity(dc_minus_1=True, consider_reload=True, peak_coef=1)
            self.solomon.push_sensor(service_name='market-checkouter',
                                     sensor_name=rps_capacity_sensor,
                                     value=maxRps, ts=rps_capacity_ts)
            self.solomon.push_sensor(service_name='market-checkouter',
                                     sensor_name=rps_capacity_dc_1_sensor,
                                     value=maxRpsDcMinus1, ts=rps_capacity_ts)
        current_instance_rps_capacity = instance_capacity_list[-1]
        return current_instance_rps_capacity

    def calc_hardware(self,
                      services,
                      dc_names,
                      rps_forecast_sensor,
                      current_instance_rps_capacity,
                      sensor_prefix):
        log.info('starting calc_hardware for "{}"'.format(sensor_prefix))
        from lib.Service import Service, RTC
        from functools import partial

        if current_instance_rps_capacity is None:
            log.warning('No current_instance_rps_capacitycurrent_instance_rps_capacity!')
            return

        forecast_ts_list, rps_forecast_values = self.get_forecast(rps_forecast_sensor)
        today = self.get_today_midnight_timestamp()
        cores_instances_df = RTC.get_services_cpu_cores_and_instances_count(services=services)
        cores_per_instance_by_dc = dict()
        instances_by_dc = dict()
        cores_by_dc = dict()
        for row in cores_instances_df.iterrows():
            dc_name = row[0].upper()
            cores_count = int(row[1]['cores'])
            self.solomon.push_sensor(service_name='market-checkouter',
                                     sensor_name='{}{}CoresHistory'.format(sensor_prefix, dc_name),
                                     value=cores_count, ts=today)
            self.solomon.push_sensor(service_name='market-checkouter',
                                     sensor_name='{}{}CoresDelta'.format(sensor_prefix, dc_name),
                                     value=0, ts=today)
            self.solomon.push_sensor(service_name='market-checkouter',
                                     sensor_name='{}{}CoresForecast'.format(sensor_prefix, dc_name),
                                     value=cores_count, ts=today)
            instance_count = int(row[1]['instances'])
            self.solomon.push_sensor(service_name='market-checkouter',
                                     sensor_name='{}{}Instances'.format(sensor_prefix, dc_name),
                                     value=instance_count, ts=today)
            cores_per_instance_by_dc[dc_name] = row[1]['cores_per_instance']
            instances_by_dc[dc_name] = instance_count
            cores_by_dc[dc_name] = cores_count

        cores_total = sum(instances_by_dc.values())
        self.solomon.push_sensor(service_name='market-checkouter',
                                 sensor_name='{}instancesTotalHistory'.format(sensor_prefix),
                                 value=cores_total,
                                 ts=today)
        self.solomon.push_sensor(service_name='market-checkouter',
                                 sensor_name='{}instancesTotalForecast'.format(sensor_prefix),
                                 value=cores_total,
                                 ts=today)

        cores_total = sum(cores_by_dc.values())
        self.solomon.push_sensor(service_name='market-checkouter',
                                 sensor_name='{}coresTotalHistory'.format(sensor_prefix),
                                 value=cores_total,
                                 ts=today)
        self.solomon.push_sensor(service_name='market-checkouter',
                                 sensor_name='{}coresTotalForecast'.format(sensor_prefix),
                                 value=cores_total,
                                 ts=today)

        get_rtc_config = partial(RTC.get_services_instances_count, services=services)
        service = Service(name="service", current_rps=1, dead_instances_per_dc_max=2)
        for dc_name in dc_names:
            service.add_instances(dc=dc_name, instances_count=get_rtc_config(dc=dc_name),
                                  instance_rps_limit=current_instance_rps_capacity)

        for row in range(len(forecast_ts_list)):
            forecast_ts = self.timestamp_to_midnight(forecast_ts_list[row])
            rps_forecast = rps_forecast_values[row]
            grow = service.calc_grow_by_rps(future_max_rps=rps_forecast,
                                            dc_minus_1=True,
                                            peak_coef=1,
                                            consider_reload=True,
                                            align_deltas=True)
            log.debug(str(grow))
            grow = grow.iloc[0]

            cores_total_forecast = int(
                sum(map(lambda n: cores_per_instance_by_dc[n] * grow[n], dc_names)))
            self.solomon.push_sensor(service_name='market-checkouter',
                                     sensor_name='{}coresTotalForecast'.format(sensor_prefix),
                                     value=cores_total_forecast,
                                     ts=forecast_ts)

            self.solomon.push_sensor(service_name='market-checkouter',
                                     sensor_name='{}instancesTotalForecast'.format(sensor_prefix),
                                     value=int(sum(map(lambda n: grow[n], dc_names))),
                                     ts=forecast_ts)
            for dc_name in dc_names:
                cores_per_instance = cores_per_instance_by_dc[dc_name]
                self.solomon.push_sensor(service_name='market-checkouter',
                                         sensor_name='{}{}CoresForecast'.format(sensor_prefix, dc_name),
                                         value=int(cores_per_instance * grow[dc_name]),
                                         ts=forecast_ts)
                self.solomon.push_sensor(service_name='market-checkouter',
                                         sensor_name='{}{}CoresDelta'.format(sensor_prefix, dc_name),
                                         value=int(cores_per_instance * grow[dc_name + '_delta']),
                                         ts=forecast_ts)

    def calc_checkouter_orders_capacity(self):
        log.info('calculating checkouter orders capacity')
        date_from = datetime.now() - timedelta(days=30)
        timestamps, checkouter_max_rps_values = self.solomon.get_values(date_from=date_from,
                                                                        service_name='market-checkouter',
                                                                        sensor_name=CHECKOUTER_RPS_CAPACITY_SENSOR)

        if len(timestamps) == 0:
            raise Exception('Cannot calculate checkouter orders: no data in "{}"'
                            .format(CHECKOUTER_RPS_CAPACITY_SENSOR))
        from sklearn.linear_model import LinearRegression as LR
        checkouter_formula_rps = LR().fit(self.checkouter_df[['rps']].values,
                                          self.checkouter_df[['orders_count']].values)
        corr = self.checkouter_df['rps'].corr(self.checkouter_df['orders_count'])
        log.debug('rps/orders corr: {}'.format(corr))

        for i in range(len(timestamps)):
            ts = self.timestamp_to_midnight(timestamps[i])
            checkouter_rps_capacity = checkouter_max_rps_values[i]
            current_max_orders = int(checkouter_formula_rps.predict([[checkouter_rps_capacity]]))
            self.solomon.push_sensor('market-checkouter', CHECKOUTER_ORDERS_CAPACITY_SENSOR, current_max_orders,
                                     ts)

        timestamps_dc1, checkouter_max_rps_values_dc1 = \
            self.solomon.get_values(date_from=date_from,
                                    service_name='market-checkouter',
                                    sensor_name=CHECKOUTER_RPS_CAPACITY_DC_1_SENSOR)

        for i in range(len(timestamps_dc1)):
            ts = self.timestamp_to_midnight(timestamps_dc1[i])
            checkouter_rps_capacity_dc1 = checkouter_max_rps_values_dc1[i]
            current_max_orders_dc1 = int(checkouter_formula_rps.predict([[checkouter_rps_capacity_dc1]]))
            self.solomon.push_sensor('market-checkouter', CHECKOUTER_ORDERS_CAPACITY_DC_1_SENSOR,
                                     current_max_orders_dc1, ts)

    def calc_carter_dau_capacity(self):
        log.info('calculating carter DAU capacity')
        date_from = datetime.now() - timedelta(days=30)
        timestamps, carter_max_rps_values = self.solomon.get_values(date_from=date_from,
                                                                    service_name='market-checkouter',
                                                                    sensor_name=CARTER_RPS_CAPACITY_SENSOR)

        if len(timestamps) == 0:
            raise Exception('Cannot calculate carter DAU capacity: no data in "{}"'.format(CARTER_RPS_CAPACITY_SENSOR))

        from sklearn.linear_model import LinearRegression as LR
        carter_formula_rps = LR().fit(self.carter_df[['rps']].values, self.carter_df[['DAU']].values)
        corr = self.carter_df['rps'].corr(self.carter_df['DAU'])
        log.debug('rps/DAU corr: {}'.format(corr))

        ts = self.timestamp_to_midnight(timestamps[-1])
        carter_rps_capacity = carter_max_rps_values[-1]
        current_max_dau = int(carter_formula_rps.predict([[carter_rps_capacity]]))
        self.solomon.push_sensor('market-checkouter', CARTER_DAU_CAPACITY_SENSOR, current_max_dau,
                                 ts)
        timestamps_dc1, carter_max_rps_values_dc1 = self.solomon.get_values(date_from=date_from,
                                                                            service_name='market-checkouter',
                                                                            sensor_name=CARTER_RPS_CAPACITY_DC_1_SENSOR)

        ts = self.timestamp_to_midnight(timestamps_dc1[-1])
        carter_rps_capacity_dc1 = carter_max_rps_values_dc1[-1]
        current_max_dau_dc1 = int(carter_formula_rps.predict([[carter_rps_capacity_dc1]]))
        log.debug('current_max_dau_dc1={}'.format(current_max_dau_dc1))
        self.solomon.push_sensor('market-checkouter', CARTER_DAU_CAPACITY_DC_1_SENSOR, current_max_dau_dc1,
                                 ts)

    def timestamp_to_midnight(self, timestamp):
        return datetime.timestamp(datetime.combine(datetime.fromtimestamp(timestamp), datetime.min.time()))

    def make_carter_forecast(self):
        log.info('making carter forecast')
        self.init_carter_df()

        from sklearn.linear_model import LinearRegression as LR
        formula_dau = LR().fit(self.carter_df[['DAU']].values, self.carter_df[['rps']].values)
        for dt in DAU_FORECAST:
            timestamp = datetime.timestamp(datetime.strptime(dt, '%Y-%m-%d'))
            dau = DAU_FORECAST.get(dt)
            rps_forecast = int(formula_dau.predict([[dau]]))
            self.solomon.push_sensor('market-checkouter', CARTER_RPS_FORECAST_SENSOR, rps_forecast, timestamp)
            self.solomon.push_sensor('market-checkouter', CARTER_DAU_FORECAST_SENSOR, dau, timestamp)
        return self.carter_df

    def init_carter_df(self):
        log.info('init_carter_df')
        from lib.tools import today
        date_start = today(minus_days=7, astype=str)
        date_end = today(minus_days=1, astype=str)
        carter_df, max_corr, q_max = self.create_carter_df(date_start, date_end)
        log.info('quantile: {}\ncorrelation: {}'.format(q_max, max_corr))
        if max_corr < 0.6:
            log.error("Correlation is too low: {}".format(max_corr))
            log.error("Falling back to dates with good correlation: {}".format(max_corr))
            carter_df, max_corr, q_max = self.create_carter_df('2021-01-18', '2021-01-26')
        self.carter_df = carter_df

    def create_carter_df(self, date_start, date_end):
        from lib.DAU import DAU
        from lib.Graphite import Graphite

        market_DAU = DAU(oauth_token=self.yql_token).get(service='white', date_from=date_start, date_to=date_end) \
            .loc['total']
        carter_df = None
        max_corr = 0
        q_max = 0
        for i in range(80, 99):
            q = i / 100
            rps_df = Graphite.get(target=CARTER_RPS_ALL_KEY, date_from=date_start, date_to=date_end)[['ts', 'value']] \
                .groupby(pd.Grouper(key='ts', freq='1d')) \
                .quantile(q)[['value']] \
                .rename(columns={'value': 'rps'})
            _df = rps_df.join(market_DAU).dropna()
            corr = _df['DAU'].corr(_df['rps'])
            if corr > max_corr:
                carter_df = _df
                max_corr = corr
                q_max = q
        return carter_df, max_corr, q_max

    def make_checkouter_forecast(self):
        log.info('making checkouter forecast')
        from datetime import datetime
        correlation = self.checkouter_df['orders_count'].corr(self.checkouter_df['rps'])
        log.info('correlation: {}'.format(correlation))
        if correlation < 0.7:
            raise Exception('correlation orders_count/rps is too low: {}'.format(correlation))
        from sklearn.linear_model import LinearRegression as LR
        formula = LR().fit(self.checkouter_df[['orders_count']].values, self.checkouter_df[['rps']].values)
        for dt in ORDERS_FORECAST:
            timestamp = datetime.timestamp(datetime.strptime(dt, '%Y-%m-%d'))
            dau = ORDERS_FORECAST.get(dt)
            rps_forecast = int(formula.predict([[dau]]))
            self.solomon.push_sensor('market-checkouter', CHECKOUTER_RPS_FORECAST_SENSOR, rps_forecast,
                                     timestamp)
            self.solomon.push_sensor('market-checkouter', CHECKOUTER_ORDERS_FORECAST_SENSOR, dau, timestamp)

    def init_checkouter_df(self):
        log.info('init_checkouter_df')
        from lib.Graphite import Graphite
        from lib.tools import today
        date_start = today(minus_days=15, astype=str)
        date_end = today(minus_days=1, astype=str)
        self.checkouter_df = \
            Graphite.get(target=CHECKOUTER_RPS_ALL_KEY, date_from=date_start, date_to=date_end)[['ts', 'value']] \
                .groupby(pd.Grouper(key='ts', freq='1d')) \
                .quantile(.95)[['value']] \
                .rename(columns={'value': 'rps'}) \
                .join(
                (
                    self.get_orders_history_df(date_start, date_end)
                )
            ).dropna() \
                .groupby(pd.Grouper(freq='2d')) \
                .sum()

    def calc_min_capacity(self):
        log.info('calculating minimal Market capacity')
        date_start = datetime.now() - timedelta(days=3)
        orders_cap_ts_list, orders_cap_values = self.solomon.get_values(date_from=date_start,
                                                                        service_name='market-checkouter',
                                                                        sensor_name=CHECKOUTER_ORDERS_CAPACITY_DC_1_SENSOR)
        checkouter_orders_capacity = dict(zip(map(self.timestamp_to_midnight, orders_cap_ts_list), orders_cap_values))
        log.debug('orders capacity dict: {}'.format(checkouter_orders_capacity))
        dau_cap_ts_list, dau_cap_values = self.solomon.get_values(date_from=date_start,
                                                                  service_name='market-checkouter',
                                                                  sensor_name=CARTER_DAU_CAPACITY_DC_1_SENSOR)
        carter_dau_capacity = dict(zip(map(self.timestamp_to_midnight, dau_cap_ts_list), dau_cap_values))
        for ts in checkouter_orders_capacity:
            conversion = self.get_conversion_for_timestamp(ts)
            log.debug('conversion: {}'.format(conversion))
            carter_dau = carter_dau_capacity.get(ts)
            if carter_dau is None or conversion is None:
                continue
            carter_orders = carter_dau * conversion
            checkouter_orders = checkouter_orders_capacity.get(ts)
            log.debug('carter orders {}, checkouter orders {}'.format(carter_orders, checkouter_orders))
            min_orders_capacity = min(checkouter_orders, carter_orders)
            self.solomon.push_sensor('market-checkouter', MIN_MARKET_ORDERS_CAPACITY_SENSOR, min_orders_capacity, ts)
            min_dau_capacity = min_orders_capacity / conversion
            self.solomon.push_sensor('market-checkouter', MIN_MARKET_DAU_CAPACITY_SENSOR, min_dau_capacity, ts)

    @staticmethod
    def get_conversion_for_timestamp(ts):
        log.info('getting conversion for ts {}'.format(ts))
        date = datetime.fromtimestamp(ts).replace(day=1).strftime('%Y-%m-%d')
        orders_forecast = ORDERS_FORECAST.get(date)
        dau_forecast = DAU_FORECAST.get(date)
        if orders_forecast is None or dau_forecast is None:
            log.info('skipping date {}'.format(date))
            return None
        else:
            return orders_forecast / dau_forecast


if __name__ == '__main__':
    log.basicConfig(level=log.DEBUG)
    import os

    yql_token = os.getenv('YQL_TOKEN')
    solomon_token = os.getenv('SOLOMON_TOKEN')
    ShootingsProcessor(solomon_token, yql_token).process()
    # ShootingsProcessor(solomon_token, yql_token).calc_min_capacity()
