import json
import logging as log
import math
import time
from datetime import datetime, timedelta

import requests

INTERVAL_DAYS = 7


class SolomonClient(object):
    def __init__(self,
                 solomon_api_url,
                 oauth_token,
                 project_id,
                 cluster_name):
        self.solomon_api_url = solomon_api_url
        self.oauth_token = oauth_token
        self.project_id = project_id
        self.cluster_name = cluster_name

    def push_sensor(self, service_name, sensor_name, value, ts):
        log.info('Pushing sensor "{}" with value {}'.format(sensor_name, value))
        log.info('Project id "{}", service name "{}", cluster_name "{}"'.format(self.project_id, service_name,
                                                                                self.cluster_name))
        if ts is None:
            ts = int(time.time())
        body = self.__make_data(sensor_name, value, ts)
        log.debug('data to push: ' + body)
        response = requests.post(
            self.solomon_api_url + '/api/v2/push?project={project}&service={service}&cluster={cluster}'.format(
                project=self.project_id, service=service_name, cluster=self.cluster_name
            ), headers=self.__make_headers(), data=body)
        if response.status_code != 200:
            raise Exception('Solomon responded with {} {} {}'.format(response.status_code, response.reason,
                                                                     response.text))

    def get_values(self, date_from, service_name, sensor_name, smooth=False, date_to=None):
        if date_to is None:
            date_to = datetime.now()

        if not smooth:
            result = [[], []]
            intervals = self.split_intervals(date_from, date_to, INTERVAL_DAYS)
            for intv in intervals:
                start = intv[0]
                end = intv[1]
                log.info('Getting data for interval: {} - {}'.format(start, end))
                ts, vals = self.__get_values(start, end, service_name, sensor_name, smooth)
                result[0] += ts
                result[1] += vals
            return result
        else:
            return self.__get_values(date_from, date_to, service_name, sensor_name, smooth)

    def __get_values(self, date_from, date_to, service_name, sensor_name, smooth=False):
        headers = {
            'Content-Type': 'application/json',
            'Authorization': 'OAuth {}'.format(self.oauth_token),
            'Accept': 'application/json'
        }

        program = '{{cluster="stable", service="{}", sensor="{}"}}'.format(service_name, sensor_name)
        if smooth:
            program = 'moving_avg({}, 5d)'.format(program)
        url = self.solomon_api_url + '/api/v2/projects/market-checkout/sensors/data'
        data = {
            "downsampling": {
                "disabled": not smooth
            },
            "forceCluster": "",
            "from": self.__format_datetime(date_from),
            "program": program,
            "to": self.__format_datetime(date_to),
        }

        log.debug('solomon get values data: {}'.format(data))
        res = requests.post(url, data=json.dumps(data), headers=headers)
        if res.status_code != 200:
            raise Exception(
                'Failed to get data. Solomon responded: {} {}'.format(res.status_code, res.text))
        log.debug('Solomon values response: {}'.format(res.text))
        response_data = json.loads(res.text)
        try:
            values = response_data['vector'][0]['timeseries']['values']
            timestamps = response_data['vector'][0]['timeseries']['timestamps']
            timestamps = list(map(lambda ts: ts / 1000, timestamps))
            return timestamps, values
        except (IndexError, KeyError):
            log.exception('Error getting values:')
            return [], []

    @staticmethod
    def __format_datetime(dt):
        return dt.strftime('%Y-%m-%dT%H:%M:%S.%f')[:-3] + 'Z'

    @staticmethod
    def split_intervals(date_from, date_to, interval_days):
        log.debug('split_intervals')
        log.debug('date_from: {}'.format(date_from))
        log.debug('date_to: {}'.format(date_to))
        intervals_count = ((date_to - date_from) / interval_days) / timedelta(days=1)
        log.debug('intervals_count: ' + str(intervals_count))
        previous_date_to = date_from
        intervals = []
        for i in range(1, math.ceil(intervals_count)):
            new_date_to = date_from + timedelta(days=interval_days) * i
            intervals.append([previous_date_to, new_date_to])
            previous_date_to = new_date_to
        intervals.append([previous_date_to, date_to])
        return intervals

    def __make_headers(self):
        return {
            'Authorization': 'OAuth ' + self.oauth_token,
            'Content-Type': 'application/json'
        }

    @staticmethod
    def __make_data(sensor_label, value, ts):
        return json.dumps({
            "sensors": [{
                "labels": {"sensor": sensor_label},
                "ts": ts,
                "value": value
            }]
        })


def main():
    log.basicConfig(level=log.DEBUG)
    import os
    oauth_token = os.getenv('SOLOMON_TOKEN')

    solomon = SolomonClient(solomon_api_url='http://solomon.yandex.net',
                            oauth_token=oauth_token,
                            project_id='market-checkout',
                            cluster_name='stable'
                            )

    from datetime import datetime, timedelta
    ts, values = solomon.get_values(datetime.now() - timedelta(days=5), 'market-checkouter', 'dailyOrdersCount',
                                    smooth=False)
    print(len(ts))
    print(ts[-1])
    print(values[-1])


if __name__ == '__main__':
    main()
