#! /usr/bin/env python
# ya make --target-platform=default-linux-x86_64
# https://sandbox.yandex-team.ru/oauth/

import codecs
import csv
import logging as log
import re

from sandbox import sdk2
from sandbox.projects.common.binary_task import deprecated as binary_task

from sandbox.projects.market.checkout.helpers import RTC
from sandbox.projects.market.checkout.helpers import YQL
from sandbox.projects.market.checkout.resources import LoyaltyLoadEndpointsRps

from urllib3.exceptions import ConnectTimeoutError


def filter_out_admin_rtc(loyalty_prod):
    return list(filter(re.compile('loyalty_(?!admin)').search, loyalty_prod))


class MarketLoyaltyLoadCalculateRps(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        yql_secret = sdk2.parameters.YavSecretWithKey(
            'YQL token secret',
            default='sec-01fwrngf9mpbggv2p2q9mb5k2e#market.loyalty.yql.oauth.token',
            required=True
        )

        with sdk2.parameters.Output:
            prod_to_load_instance_count_ratio = sdk2.parameters.Float('prod-to-load instance count ratio',
                                                                      default_value='')
            prod_to_load_instance_power_ratio = sdk2.parameters.Float('prod-to-load instance power ratio',
                                                                      default_value='')
            endpoints_rps = sdk2.parameters.String('Endpoints and RPS (csv)', default_value='')

    def on_execute(self):
        log.info('Starting on_execute')
        binary_task.LastBinaryTaskRelease.on_execute(self)
        yql_token = self.Parameters.yql_secret.value()
        rps_query_resource_path = \
            'sandbox/projects/market/checkout/MarketLoyaltyLoadCalculateRps/loyalty_rps_request.sql'
        current_rps_total, rps_ratio = self.get_total_rps_and_ratios(yql_token, rps_query_resource_path)
        log.info('Total current RPS: ' + str(current_rps_total))
        log.info('RPS ratio result: ' + str(rps_ratio))
        try:
            instance_count_ratio, instance_power_ratio = self.get_prod_to_load_ratios()
            self.Parameters.prod_to_load_instance_count_ratio = instance_count_ratio
            self.Parameters.prod_to_load_instance_power_ratio = instance_power_ratio
        except (ValueError, ConnectTimeoutError) as err:
            log.error(str(err))
            log.debug("Cannot get nanny service info about instances", exc_info=True)
            # hardcoded data
            self.Parameters.prod_to_load_instance_count_ratio = 29
            self.Parameters.prod_to_load_instance_power_ratio = (2 * 8 + 2 * 16) / (29 * 8 + 29 * 16)
        self.publish_output_resource(rps_ratio)

    @staticmethod
    def get_total_rps_and_ratios(yql_token, rps_query_resource_path):
        rows = YQL.query_clickhouse(yql_token, rps_query_resource_path)
        log.debug('got rps rows: {}'.format(rows))
        sum_rps = sum(map(lambda r: float(r[1]), rows))
        sum_parts = 0
        log.info('current RPS total:' + str(sum_rps))
        rps_ratio = []
        for row in rows:
            page_id = row[0]
            rps = float(row[1])
            rps_part = rps / sum_rps
            sum_parts += rps_part
            rps_ratio.append([page_id, rps_part])
        log.debug('sum of all ratios:' + str(sum_parts))
        return sum_rps, rps_ratio

    @staticmethod
    def get_prod_to_load_ratios():
        prod_category_prefix = 'market_pers_loyalty'
        prod_service_prefix = 'production_market_loyalty'
        prod_services = filter_out_admin_rtc(
            RTC.get_services(category_prefix=prod_category_prefix, service_prefix=prod_service_prefix)
        )

        prod_rtc_data = RTC.get_service_rtc_data(
            category_prefix=prod_category_prefix,
            service_prefix=prod_service_prefix,
            service_filter=filter_out_admin_rtc)

        log.info('Prod config: %f cores, %f memory' % prod_rtc_data)
        dc_minus_1_prod_instance_count = RTC.get_dc_minus_1_prod_instance_count(services=prod_services,
                                                                                category_prefix=prod_category_prefix)
        log.info('DC-1 PROD instance count: {}'.format(dc_minus_1_prod_instance_count))

        load_rtc_data = RTC.get_service_rtc_data(category_prefix='market_marketloyalty_market-loyalty-load',
                                                 service_prefix='testing_market_market_loyalty_load')
        log.info('Load config: %f cores, %f memory' % load_rtc_data)

        instance_power_ratio = min(1 / (prod_rtc_data[0] / load_rtc_data[0]), 1 / (prod_rtc_data[1] / load_rtc_data[1]))
        log.info('instance_power_ratio= ' + str(instance_power_ratio))
        return dc_minus_1_prod_instance_count, instance_power_ratio

    def publish_output_resource(self, rps_ratio):
        data = sdk2.ResourceData(LoyaltyLoadEndpointsRps(self, 'Loyalty load endpoints RPS', 'endpoints_rps.csv'))
        with codecs.open('endpoints_rps.csv', 'w', encoding='utf-8') as f:
            writer = csv.writer(f)
            for endpoint_rps in rps_ratio:
                writer.writerow(endpoint_rps)
        with open('endpoints_rps.csv', 'r') as f:
            self.Parameters.endpoints_rps = f.read()
        data.ready()
