#! /usr/bin/env python

import codecs
import csv
import logging as log
import random

from sandbox import sdk2
from sandbox.projects.common.binary_task import deprecated as binary_task

from sandbox.projects.market.checkout.MarketLoyaltyLoadCalculateRps import MarketLoyaltyLoadCalculateRps
from sandbox.projects.market.checkout.MarketLoyaltyLoadGenerateAmmo.complex_load import capacity_load_gens
from sandbox.projects.market.checkout.resources import LoyaltyLoadAmmo
from sandbox.projects.market.checkout.resources import LoyaltyLoadConfig
from sandbox.projects.market.checkout.resources import LoyaltyLoadEndpointsRps

MAX_AMMO_COUNT = 1000000
DEFAULT_AUTO_STOP_CONFIG = 'quantile(99,700,3s) http(4xx,30%,10) http(50x,10%,10) http(51x,10%,10) net(xx,10%,10)'


def load_api_methods(rps_array):
    result = []
    for row in rps_array:
        log.debug('reading row: ' + str(row))
        result.append({
            "page_id": row[0],
            "rps_multiplier": row[1]
        })
    return result


class MarketLoyaltyLoadGenerateAmmo(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        imbalance_shooting = sdk2.parameters.Bool(
            'Shoot until service is unstable?',
            description='When checked "line" tank rps schedule will be used until service fails. '
                        'When unchecked then rps will be constant ( "const" schedule )', default=True
        )
        yql_secret = sdk2.parameters.YavSecretWithKey(
            'YQL token secret',
            default='sec-01fwrngf9mpbggv2p2q9mb5k2e#market.loyalty.yql.oauth.token',
            required=True
        )
        with imbalance_shooting.value[False]:
            test_duration_minutes = sdk2.parameters.Integer('Test duration minutes', default=10)
            total_rps = sdk2.parameters.Integer('Total rps', default=100)

        with imbalance_shooting.value[True]:
            start_rps = sdk2.parameters.Integer('Start RPS', default=20)
            target_rps = sdk2.parameters.Integer('Target RPS', default=80)
            rps_increase_step = sdk2.parameters.Integer('RPS increase step', default=20)
            rps_step_duration_seconds = sdk2.parameters.Integer('RPS step duration seconds', default=100)
            autostop_config = sdk2.parameters.String('Shooting autostop config',
                                                     default=DEFAULT_AUTO_STOP_CONFIG)

        endpoints_rps_resource = sdk2.parameters.Resource('Endpoints RPS resource',
                                                          resource_type=LoyaltyLoadEndpointsRps)
        custom_endpoints_rps = sdk2.parameters.String('Custom endpoints RPS', multiline=True)

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)
        self.generate_ammo()
        self.generate_config()

    def generate_ammo(self):
        meta = self.generate_meta()
        log.info('generating ammo')
        max_rounds = MAX_AMMO_COUNT
        if self.Parameters.imbalance_shooting:
            total_rounds = min(max_rounds, self.calculate_total_rounds_for_imbalance_shooting())
        else:
            total_rounds = min(max_rounds, self.Parameters.total_rps * self.Parameters.test_duration_minutes * 60)
        log.info('total rounds: ' + str(total_rounds))

        ammo = []
        total_count = 0
        for meta_line in meta:
            page_id = meta_line["page_id"]
            generator_with_params = meta_line["generator"]
            rps_multiplier = float(meta_line["rps_multiplier"])

            count = max(int(rps_multiplier * total_rounds), 1)
            log.info('count for {} = {}'.format(page_id, count))
            total_count += count
            method_gen, params = generator_with_params

            for i in range(0, count):
                if params is None:
                    ammo.append(method_gen())
                else:
                    ammo.append(method_gen(params))
        log.info('total request count: ' + str(total_count))

        random.shuffle(ammo)

        data = sdk2.ResourceData(LoyaltyLoadAmmo(self, 'Loyalty load ammo', 'ammo.txt'))
        with codecs.open('ammo.txt', 'w', encoding='utf-8') as f:
            for single_ammo in ammo:
                f.write(single_ammo)
        data.ready()

    def generate_meta(self):
        rps_array = []
        log.info('Endpoints rps resource = `{}` and custom endpoints rps = `{}`',
                 self.Parameters.endpoints_rps_resource,
                 self.Parameters.custom_endpoints_rps)

        if bool(self.Parameters.endpoints_rps_resource) or bool(self.Parameters.custom_endpoints_rps):
            rps_lines = self.get_endpoints_rps_lines()
            rps_array = csv.reader(rps_lines, delimiter=',', quotechar='"')
        else:
            sum_rps, rps_array = MarketLoyaltyLoadCalculateRps.get_total_rps_and_ratios(
                self.Parameters.yql_secret.value(),
                'sandbox/projects/market/checkout/MarketLoyaltyLoadCalculateRps/loyalty_rps_request.sql'
            )

        api_methods = load_api_methods(rps_array)
        meta = []

        for api_method in api_methods:
            page_id = api_method["page_id"]

            if page_id not in capacity_load_gens:
                log.warning('Attention! Method not supported: %s' % page_id)
                continue

            rps_multiplier = float(api_method["rps_multiplier"])

            meta.append({
                'page_id': page_id,
                'rps_multiplier': rps_multiplier,
                'generator': capacity_load_gens[page_id]
            })

        return meta

    def get_endpoints_rps_lines(self):
        is_custom_shooting = bool(self.Parameters.custom_endpoints_rps)
        if is_custom_shooting:
            return self.Parameters.custom_endpoints_rps.splitlines()
        else:
            rps_resource_path = str(sdk2.ResourceData(self.Parameters.endpoints_rps_resource).path)
            log.debug('opening endpoints_rps_resource: ' + rps_resource_path)
            with open(rps_resource_path) as rps_file:
                return rps_file.readlines()

    def calculate_total_rounds_for_imbalance_shooting(self):
        total_rounds = 0
        rps = self.Parameters.start_rps
        while rps <= self.Parameters.target_rps:
            total_rounds += rps * self.Parameters.rps_step_duration_seconds
            rps += self.Parameters.rps_increase_step
        return total_rounds

    def generate_config(self):
        config_file_name = 'check.yaml'
        config_resource = sdk2.ResourceData(LoyaltyLoadConfig(self, 'Loyalty load config', config_file_name))

        config = self.read_template()
        if self.Parameters.imbalance_shooting:
            config = self.generate_config_lines_for_imbalance_shooting(config)
        else:
            config = self.generate_config_lines_for_const_shooting(config)

        with codecs.open(config_file_name, 'w', encoding='utf-8') as f:
            f.write(config)
        config_resource.ready()

    def read_template(self):
        from library.python import resource
        return resource.find('sandbox/projects/market/checkout/MarketLoyaltyLoadGenerateAmmo/config/check.yaml.tmpl')

    def generate_config_lines_for_const_shooting(self, config):
        log.info('Adding config lines for CONST shooting')
        rps = int(self.Parameters.total_rps)
        start_rps = max(int(rps * 0.05), 1)
        rps_schedule = 'const({}, 5m) line({},{},2m) const({}, {}m)' \
            .format(start_rps, start_rps, rps, rps, self.Parameters.test_duration_minutes)
        return config.format(schedule=rps_schedule, autostop_enabled='false',
                             autostop_config='')

    def generate_config_lines_for_imbalance_shooting(self, config):
        log.info('Adding config lines for IMBALANCE shooting')
        rps_schedule = 'const(1, 10s) step({}, {}, {}, {}s)'.format(self.Parameters.start_rps,
                                                                    self.Parameters.target_rps,
                                                                    self.Parameters.rps_increase_step,
                                                                    self.Parameters.rps_step_duration_seconds)
        log.info('Using autostop config: ' + str(self.Parameters.autostop_config))
        autostop_config = '- ' + '\n  - '.join(self.Parameters.autostop_config.split(' '))
        return config.format(schedule=rps_schedule, autostop_enabled='true',
                             autostop_config=autostop_config)
