# -*- coding: utf8 -*-
import time
from datetime import datetime
import sched
from time import time as time_obj
import random
import logging as log

INSERT_ORDER_EVENT_SQL = '''INSERT INTO
public.order_event (history_id, event_id, host, trantime, event_type, bucket, order_id, publish_time) VALUES '''
ORDER_EVENT_INSERT_VALUES = '''({history_id}, {event_id}, 'shoot-notifier-script', '{time}', {event_type},
{bucket}, {order_id}, '{time}')'''
INSERT_ORDER_HISTORY_SQL = '''INSERT INTO public.order_history (id, event_type, author_role, author_id, from_dt, to_dt,
prev_history_id, order_id, status, substatus, delivery_id, total, buyer_total, payment_id, balance_order_id,
refund_planned, refund_actual, refund_id, refund_history_id, author_uid, author_shop_id, subsidy_id,
subsidy_balance_order_id, subsidy_refund_planned, subsidy_refund_actual, is_archived, user_id,
no_auth, receipt_id, read_by_user, items_total, buyer_items_total, subsidy_total, fee_total, event_id, host, trantime,
 cancellation_requested, shop_id, return_id, publish_time, request_id, reason, initiator_type, initiator_id)
    VALUES '''
ORDER_HISTORY_INSERT_VALUES = '''({history_id}, {event_type}, 0, 0, '{time}', null, 120714117, {order_id}, 3, 29,
53494530, 8147.00, 8147.00, null, null, null, null, null, null, 0, 0, null, null, null, null, false, 151731, 0, null,
false, 8147.00, 8147.00, 6.00, 8.15, {event_id}, 'shoot-notifier-script', '{time}', false, 431782, null, null,
'shoot-notifier-script', null, 0, null)'''
INSERT_ORDER_PROPERTY_SQL = u'''INSERT INTO public.order_property (order_id, name, text_value)
VALUES ({order_id}, 'random_property{property_index}',
        '{property_text}');'''
BUCKETS = range(60)
# ORDER_IDS_FOR_DIFFERENT_CONSUMERS = (7816455, 7816451, 7816452, 7816453, 7816454)

# Для получения вероятности событий
SELECT_EVENT_TYPES = '''select (count(1) / 300000::float) prc, event_type  from (
select * from order_history order by id desc
limit 300000) q
group by event_type
order by prc desc'''

# Результаты из event_type_probabilities.sql
EVENT_TYPE_PROBABILITIES = {
    0.224306666666667: 10,
    0.17168: 1,
    0.14712: 2,
    0.0802133333333333: 27,
    0.0502833333333333: 23,
    0.04504: 34,
    0.0383766666666667: 0,
    0.03813: 11,
    0.0379266666666667: 48,
    0.0327433333333333: 18,
    0.0299466666666667: 3,
    0.0245733333333333: 45,
    0.02315: 30,
    0.0123266666666667: 40,
    0.01133: 41,
    0.00966: 35,
    0.00558: 14,
    0.00390666666666667: 24,
    0.00373666666666667: 37,
    0.00325333333333333: 15,
    0.00228: 7,
    0.0006633333333333: 22,
    0.0005866666666667: 4,
    0.00042: 44,
    0.0004166666666667: 12,
    0.0003933333333333: 5,
    0.0003666666666667: 16,
    0.0003466666666667: 21,
    0.0002433333333333: 8,
    0.0002133333333333: 43,
    0.00016: 29,
    0.00012: 47,
    0.0001133333333333: 13,
    0.0001066666666667: 19,
    0.0001033333333333: 28,
    0.0000833333333333: 17,
    0.00006: 38,
    0.00004: 20
}


# in future: DB access move to Database class

class EventInserter:
    def __init__(self, connection, batch_size=11, frequency_seconds=0.1, duration_seconds=60, average_size_kb=1):
        self.connection = connection
        self.batch_size = batch_size
        self.frequency_seconds = frequency_seconds
        self.duration_seconds = int(duration_seconds)
        self.cursor = self.connection.cursor()
        self.history_id = self.__fetch_single('''select max(oh.id) from order_history oh''') + 1
        self.event_id = self.__fetch_single('''select max(oe.event_id) from order_event oe ''') + 1
        self.inserted_count = 0
        self.average_size_kb = average_size_kb
        self.order_ids = self.prepare()
        self.start_time = self.last_commit_time = time_obj()

    def do_work(self):
        log.debug("Init EventInserter")
        try:
            log.debug("Scheduler action")
            self.schedule_action(
                self.frequency_seconds,
                self.duration_seconds
            )
            log.debug("Action scheduled")
        except Exception as e:
            log.error(e)
        finally:
            self.set_sequences()
            self.connection.commit()
            log.info('Created {} events'.format(self.inserted_count))

    def insert_events(self):
        now = datetime.today().strftime('%Y-%m-%d %H:%M:%S.%f')

        history_values = []
        event_values = []
        event_type = EVENT_TYPE_PROBABILITIES[list(EVENT_TYPE_PROBABILITIES)[-1]]
        for _ in range(self.batch_size):
            rnd = random.random()
            s = 0
            for probability in EVENT_TYPE_PROBABILITIES:
                s += probability
                if rnd < s:
                    event_type = EVENT_TYPE_PROBABILITIES[probability]
                    break
            order_id = random.choice(self.order_ids)
            history_values.append(ORDER_HISTORY_INSERT_VALUES.format(time=now,
                                                                     history_id=self.history_id,
                                                                     event_id=self.event_id,
                                                                     order_id=order_id,
                                                                     event_type=event_type))
            event_values.append(ORDER_EVENT_INSERT_VALUES.format(time=now,
                                                                 history_id=self.history_id,
                                                                 event_id=self.event_id,
                                                                 bucket=random.choice(BUCKETS),
                                                                 order_id=order_id,
                                                                 event_type=event_type))
            self.history_id += 1
            self.event_id += 1

        self.__execute(INSERT_ORDER_HISTORY_SQL + ', '.join(history_values))
        self.__execute(INSERT_ORDER_EVENT_SQL + ', '.join(event_values))
        self.inserted_count += self.batch_size
        self.__commit_if_needed()

    def insert_properties(self, order_id):
        total_size = self.average_size_kb * 1024
        order_property_count = int(round(total_size / 2000))
        for i in range(order_property_count):
            property_text = random.randint(900, 1200) * 'a'
            self.__execute(INSERT_ORDER_PROPERTY_SQL.format(
                order_id=order_id, property_index=random.randint(1, 1000000), property_text=property_text))

    def __commit_if_needed(self):
        if time_obj() - self.last_commit_time > 5:
            speed = self.inserted_count / (time_obj() - self.start_time)
            log.info('Saving {} events. Speed {} e/s'.format(self.inserted_count, speed))
            self.connection.commit()
            self.last_commit_time = time_obj()

    def __execute(self, sql):
        self.cursor.execute(sql)

    def execute_and_fetch(self, sql):
        self.__execute(sql)
        return self.cursor.fetchall()

    def __fetch_single(self, sql):
        self.__execute(sql)
        return list(map(lambda r: r[0], self.cursor.fetchall()))[0]

    def set_sequences(self):
        log.info('Setting sequences', self.history_id)
        self.__execute('''SELECT setval('order_history_seq', {history_id}, true)'''.format(history_id=self.history_id))
        self.__execute('''SELECT setval('order_event_seq', {event_id}, true)'''.format(event_id=self.event_id))

    def schedule_action(self, frequency, duration, *args):
        scheduler = sched.scheduler(time.time, time.sleep)
        no_of_events = int(duration / frequency)
        priority = 1  # not used, lets you assign execution order to events scheduled for the same time
        for i in range(no_of_events):
            delay = i * frequency
            scheduler.enter(delay, priority, self.insert_events, args)
        scheduler.run()

    def prepare(self):
        self.__execute("delete from order_property")
        self.connection.commit()
        # result = self.execute_and_fetch("select o.id from orders o order by o.id desc limit 1000")
        order_ids = [7816455, 7816451, 7816452, 7816453, 7816454]
        for order_id in order_ids:
            self.insert_properties(order_id)

        self.connection.commit()

        # return list(map(lambda r: r[0], result))
        return order_ids

if __name__ == "__main__":
    from database import Database

    db = Database(
        "market_checkouter_prod",
        "market_checkouter",
        "sas-cly1hsv1shkv06r8.db.yandex.net",
        6432
    )
    password = "X2Xi0NE9eTUDHp6yWhiiS161VaCM3yLLttALiy3jd5WGbrPTu4fYxIt2hCIbsS6Vi1Z1LSjSWvFGGn6j8YirNAxBm53NrebV1ppvwaRlk7RoHr9dWuWFgeMER1xyfHp3"
    with db.connect(password) as connection:
        inserter = EventInserter(connection, batch_size=10, duration_seconds=100, average_size_kb=10)
        inserter.do_work()
