#! /usr/bin/env python
# ya make --target-platform=default-linux-x86_64
# https://sandbox.yandex-team.ru/oauth/
# ./MarketPushApiLoadCalculateRps upload --verbose --owner MARKET --attr release=stable --attr ttl=inf --attr task_type=MARKET_PUSH_API_LOAD_CALCULATE_RPS

import codecs
import csv
import logging as log

from sandbox import sdk2
from sandbox.projects.common.binary_task import deprecated as binary_task

from sandbox.projects.market.checkout.helpers import RTC
from sandbox.projects.market.checkout.helpers import YQL
from sandbox.projects.market.checkout.resources import PushApiLoadEndpointsRps


class MarketPushApiLoadCalculateRps(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        yql_token_owner = sdk2.parameters.String('YQL token OWNER in Vault', default_value='zomb-emily')
        yql_token_name = sdk2.parameters.String('YQL token NAME in Vault', default_value='YQL_TOKEN')

        with sdk2.parameters.Output:
            prod_to_load_instance_count_ratio = sdk2.parameters.Float('prod-to-load instance count ratio',
                                                                      default_value='')
            endpoints_rps = sdk2.parameters.String('Endpoints and RPS (csv)', default_value='')

    def on_execute(self):
        log.info('Starting on_execute')
        binary_task.LastBinaryTaskRelease.on_execute(self)
        yql_token = sdk2.Vault.data(self.Parameters.yql_token_owner, self.Parameters.yql_token_name)
        rps_query_resource_path = 'sandbox/projects/market/checkout/MarketPushApiLoadCalculateRps/rps_query.sql'
        current_rps_total, rps_ratio = self.current_prod_rps_and_rps_ratio(rps_query_resource_path, yql_token)
        log.info('Total current PushApi RPS: ' + str(current_rps_total))
        log.info('RPS ratio result: ' + str(rps_ratio))
        instance_count_ratio = self.get_prod_to_load_instance_count_ratio()
        self.Parameters.prod_to_load_instance_count_ratio = instance_count_ratio
        self.publish_output_resource(rps_ratio)

    def current_prod_rps_and_rps_ratio(self, rps_query_resource_path, yql_token):
        rows = YQL.query_clickhouse(yql_token, rps_query_resource_path)
        log.debug('got rps rows: {}'.format(rows))
        sum_rps = sum(map(lambda r: float(r[2]), rows))
        sum_parts = 0
        log.info('current RPS total:' + str(sum_rps))
        rps_ratio = []
        for row in rows:
            http_method = row[0]
            method = row[1]
            rps = float(row[2])

            rps_part = rps / sum_rps
            sum_parts += rps_part

            rps_ratio.append([http_method, method, rps_part])

        if 1 - sum_parts > 0.001:
            raise Exception('sum of all ratios must be 1 but was {}'.format(sum_parts))
        return sum_rps, rps_ratio

    def get_prod_to_load_instance_count_ratio(self):
        prod_services = RTC.get_services(category_prefix='market_checkout_checkouter_production',
                                         service_prefix='production_market_checkouter')
        dc_minus_1_prod_instance_count = \
            RTC.get_dc_minus_1_prod_instance_count(services=prod_services,
                                                   category_prefix='market_checkout_checkouter_production')
        log.info('DC-1 PROD instance count: {}'.format(dc_minus_1_prod_instance_count))
        return dc_minus_1_prod_instance_count

    def publish_output_resource(self, rps_config):
        data = sdk2.ResourceData(PushApiLoadEndpointsRps(self, 'PushApi load endpoints RPS', 'endpoints_rps.csv'))
        with codecs.open('endpoints_rps.csv', 'w', encoding='utf-8') as f:
            writer = csv.writer(f)
            for endpoint_rps in rps_config:
                # http_method,method,rps
                writer.writerow([endpoint_rps[0]] + [endpoint_rps[1]] + [str(endpoint_rps[2])])
        with open('endpoints_rps.csv', 'r') as f:
            self.Parameters.endpoints_rps = f.read()
        data.ready()
