#! /usr/bin/env python
# don't forget to run deploy.sh after PR was merged

import logging as log

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt
from sandbox.projects.common.binary_task import deprecated as binary_task
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.projects.tank.lunapark_client.client import LunaparkApiClient

from sandbox.projects.market.checkout.MarketCarterLoadCalculateRps import MarketCarterLoadCalculateRps
from sandbox.projects.market.checkout.MarketCarterLoadGenerateAmmo import MarketCarterLoadGenerateAmmo
from sandbox.projects.market.checkout.MarketCheckouterLoadCalculateRps import MarketCheckouterLoadCalculateRps
from sandbox.projects.market.checkout.MarketCheckouterLoadGenerateAmmoFullLoad import \
    MarketCheckouterLoadGenerateAmmoFullLoad
from sandbox.projects.market.checkout.MarketLoyaltyLoadCalculateRps import MarketLoyaltyLoadCalculateRps
from sandbox.projects.market.checkout.MarketLoyaltyLoadGenerateAmmo import MarketLoyaltyLoadGenerateAmmo
from sandbox.projects.market.checkout.MarketPushApiLoadCalculateRps import MarketPushApiLoadCalculateRps
from sandbox.projects.market.checkout.MarketPushApiLoadGenerateAmmo import MarketPushApiLoadGenerateAmmo
from sandbox.projects.market.checkout.MarketServiceLoadTest.nanny_client import NannyClient
from sandbox.projects.market.checkout.MarketServiceLoadTest.sync_properties import PropertySyncer
from sandbox.projects.market.checkout.PushToSolomon import PushToSolomon
from sandbox.projects.market.checkout.resources import CarterLoadAmmo, LoyaltyLoadAmmo, LoyaltyLoadConfig, \
    LoyaltyLoadEndpointsRps, PushApiLoadEndpointsRps, PushApiLoadAmmo, PushApiLoadConfig
from sandbox.projects.market.checkout.resources import CarterLoadConfig
from sandbox.projects.market.checkout.resources import CarterLoadEndpointsRps
from sandbox.projects.market.checkout.resources import CheckouterLoadAmmo
from sandbox.projects.market.checkout.resources import CheckouterLoadConfig
from sandbox.projects.market.checkout.resources import CheckouterLoadEndpointsRps

CHECKOUTER_DEFAULT_AUTOSTOP_CONFIG = \
    'quantile(99,2500,3s) http(4xx,10%,10) http(50x,10%,10) http(51x,10%,10) net(xx,10%,10)'
CARTER_DEFAULT_AUTOSTOP_CONFIG = \
    'quantile(99,800,3s) http(4xx,10%,10) http(50x,10%,10) http(51x,10%,10) net(xx,10%,10)'
PUSH_API_DEFAULT_AUTOSTOP_CONFIG = \
    'quantile(99,1000,3s) http(4xx,10%,10) http(50x,10%,10) http(51x,10%,10) net(xx,10%,10)'
LOYALTY_DEFAULT_AUTOSTOP_CONFIG = \
    'quantile(99,700,3s) http(4xx,30%,10) http(50x,10%,10) http(51x,10%,10) net(xx,10%,10)'

SERVICES = {
    'checkouter': {
        'name': 'checkouter',
        'nanny_service_prefix': 'testing_market_checkouter_load_',
        'dc_names': ('sas', 'iva', 'vla'),
        'calculate_rps_task': MarketCheckouterLoadCalculateRps,
        'generate_ammo_task': MarketCheckouterLoadGenerateAmmoFullLoad,
        'default_autostop_config': CHECKOUTER_DEFAULT_AUTOSTOP_CONFIG,
        'ammo_resource_type': CheckouterLoadAmmo,
        'shoot_config_resource_type': CheckouterLoadConfig,
        'rps_resource_type': CheckouterLoadEndpointsRps,
        'solomon_project_id': 'market-checkout',
        'solomon_service_name': 'market-checkouter',
    },
    'carter': {
        'name': 'carter',
        'nanny_service_prefix': 'testing_market_carter_load_',
        'dc_names': ('iva', 'vla', 'sas'),
        'calculate_rps_task': MarketCarterLoadCalculateRps,
        'generate_ammo_task': MarketCarterLoadGenerateAmmo,
        'default_autostop_config': CARTER_DEFAULT_AUTOSTOP_CONFIG,
        'ammo_resource_type': CarterLoadAmmo,
        'shoot_config_resource_type': CarterLoadConfig,
        'rps_resource_type': CarterLoadEndpointsRps,
        'solomon_project_id': 'market-checkout',
        'solomon_service_name': 'market-carter',
    },
    'loyalty': {
        'name': 'loyalty',
        'nanny_service_prefix': 'testing_market_market_loyalty_load_',
        'dc_names': ('sas', 'iva'),
        'calculate_rps_task': MarketLoyaltyLoadCalculateRps,
        'generate_ammo_task': MarketLoyaltyLoadGenerateAmmo,
        'default_autostop_config': LOYALTY_DEFAULT_AUTOSTOP_CONFIG,
        'ammo_resource_type': LoyaltyLoadAmmo,
        'shoot_config_resource_type': LoyaltyLoadConfig,
        'rps_resource_type': LoyaltyLoadEndpointsRps,
        'solomon_project_id': 'market-checkout',
        'solomon_service_name': 'market-loyalty',
    },
    'push-api': {
        'name': 'push-api',
        'nanny_service_prefix': 'testing_market_checkout_push_api_load_',
        'dc_names': ('iva', 'sas', 'vla'),
        'calculate_rps_task': MarketPushApiLoadCalculateRps,
        'generate_ammo_task': MarketPushApiLoadGenerateAmmo,
        'default_autostop_config': PUSH_API_DEFAULT_AUTOSTOP_CONFIG,
        'ammo_resource_type': PushApiLoadAmmo,
        'shoot_config_resource_type': PushApiLoadConfig,
        'rps_resource_type': PushApiLoadEndpointsRps,
        'solomon_project_id': 'market-checkout',
        'solomon_service_name': 'market-push-api',
    }
}


class MarketServiceLoadTest(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        use_custom_rps_ratio = sdk2.parameters.Bool('Use custom RPS ratio', default_value=False)
        skip_options = sdk2.parameters.Bool('Skip getting options', default_value=False)
        with use_custom_rps_ratio.value[True]:
            custom_rps_ratio = sdk2.parameters.String("Custom RPS ratio", multiline=True)
        imbalance_shooting = sdk2.parameters.Bool(
            'Shoot until service is unstable?',
            description='When checked "line" tank rps schedule will be used until service fails. '
                        'When unchecked then rps will be constant ( "const" schedule )', default=True
        )
        with imbalance_shooting.value[False]:
            test_duration_minutes = sdk2.parameters.Integer('Test duration minutes', default=10)
            total_rps = sdk2.parameters.Integer('Total rps', default=8)

        with imbalance_shooting.value[True]:
            start_rps = sdk2.parameters.Integer('Start RPS', default=5)
            target_rps = sdk2.parameters.Integer('Target RPS', default=50)
            rps_increase_step = sdk2.parameters.Integer('RPS increase step', default=1)
            rps_step_duration_seconds = sdk2.parameters.Integer('RPS step duration seconds', default=100)
            autostop_config = sdk2.parameters.String('Shooting autostop config')
        with sdk2.parameters.RadioGroup("Target service", required=True) as target_service:
            target_service.values['checkouter'] = target_service.Value(value='Checkouter', default=True)
            target_service.values['carter'] = target_service.Value(value='Carter')
            target_service.values['loyalty'] = target_service.Value(value='Loyalty')
            target_service.values['push-api'] = target_service.Value(value='Push API')
            with target_service.value['checkouter']:
                with use_custom_rps_ratio.value[False]:
                    with sdk2.parameters.Group("Archive API options") as archive_api_block:
                        shoot_only_archive_api = sdk2.parameters.Bool('Shoot only at the archive API',
                                                                      default_value=False)
                        use_archive_api_statistics = sdk2.parameters.Bool('Use archive API statistics',
                                                                          default_value=True)
        yql_token_owner = sdk2.parameters.String('YQL token OWNER in Vault', default_value='zomb-emily')
        yql_token_name = sdk2.parameters.String('YQL token NAME in Vault', default_value='YQL_TOKEN')
        nanny_token_owner = sdk2.parameters.String('NANNY token OWNER in Vault', default_value='robot-market-chkout')
        nanny_token_name = sdk2.parameters.String('NANNY token NAME in Vault', default_value='NANNY_TOKEN')
        sync_configuration_properties = sdk2.parameters.Bool('Sync properties with production?', default_value=True)
        carter_load_url = sdk2.parameters.String('Carter load url',
                                                 default_value='http://vla0-7305-vla-market-test-carte-dde-8041'
                                                               '.gencfg-c.yandex.net:8041/configuration')
        notifier_load_url = sdk2.parameters.String('Notifier load url',
                                                   default_value='http://vla2-0981-06c-vla-market-test--140-25282'
                                                                 '.gencfg-c.yandex.net:25282')
        checkouter_url = sdk2.parameters.String('Checkouter url',
                                                default='http://checkouter.load.vs.market.yandex.net:39001')
        report_url = sdk2.parameters.String('Report url',
                                            default='http://warehouse-report.blue.tst.vs.market.yandex.net:17051')
        stocks_url = sdk2.parameters.String('Stocks url', default='https://bos.tst.vs.market.yandex.net/stocks')
        loyalty_url = sdk2.parameters.String('Loyalty url',
                                             default='http://sas2-3723-sas-market-test-mark-05d-25713.gencfg-c.yandex'
                                                     '.net:25713')

    def on_execute(self):
        log.info('Starting on_execute')
        binary_task.LastBinaryTaskRelease.on_execute(self)
        self.lunaparkClient = LunaparkApiClient()
        self.target_service = SERVICES[self.Parameters.target_service]
        nanny_token = sdk2.Vault.data(self.Parameters.nanny_token_owner, self.Parameters.nanny_token_name)
        self.nanny_client = NannyClient(self, nanny_token)
        with self.memoize_stage.prepare:
            if self.Parameters.sync_configuration_properties:
              self.set_info('Syncing properties with PROD...')
              PropertySyncer(self).sync_all_properties(self.Parameters.carter_load_url,
                                                       self.Parameters.notifier_load_url)

            self.set_info('Pausing service actions...')
            for dc in self.target_service['dc_names']:
                self.nanny_client.pause_actions(self.target_service['nanny_service_prefix'] + dc)
            self.Context.imbalance_rps_list = []

        if not self.Parameters.use_custom_rps_ratio:
            with self.memoize_stage.calculate_rps:
                self.set_info('Getting endpoint RPS ratio from PROD...')
                self.calculate_rps()
            with self.memoize_stage.get_calculate_rps_results:
                log.info('calculate_rps finished')
                rps_task = self.get_task(self.Context.calculate_rps_task_id)
                self.Context.prod_to_load_instance_count_ratio = rps_task.Parameters.prod_to_load_instance_count_ratio
                self.Context.prod_to_load_instance_power_ratio = 1

        for dc_name in self.target_service['dc_names']:
            self.shoot_dc(dc_name)

        with self.memoize_stage.get_shoot_results:
            self.set_info('Resuming service actions...')
            for dc in self.target_service['dc_names']:
                self.nanny_client.resume_actions(self.target_service['nanny_service_prefix'] + dc)
            all_rps = self.Context.imbalance_rps_list
            self.set_info('Shootings finished. Imbalance RPS: {}'.format(all_rps))
            avg_imbalance_rps = sum(all_rps) / len(all_rps)
            self.set_info('Average imbalance RPS: {}'.format(avg_imbalance_rps))

            if not self.Parameters.use_custom_rps_ratio:
                prod_max_rps = int(avg_imbalance_rps * self.Context.prod_to_load_instance_count_ratio)
                self.set_info("Prod max RPS: {}".format(prod_max_rps))
                prod_instance_max_rps = int(avg_imbalance_rps)
                self.set_info("Prod instance max RPS: {}".format(prod_instance_max_rps))
                with self.memoize_stage.push_max_rps:
                    push_task_ids = [self.push_to_solomon('maxInstanceRps', prod_instance_max_rps)]

                    raise sdk2.WaitTask(push_task_ids, ctt.Status.Group.SUCCEED, wait_all=True, timeout=3600)

    def shoot_dc(self, dc_name):
        with self.memoize_stage['generate_ammo_' + dc_name]:
            self.set_info('Generating Ammo for {}...'.format(dc_name))
            self.generate_ammo()
        with self.memoize_stage['shoot_' + dc_name]:
            config_resource = self.get_task_resource(self.Context.generate_ammo_task_id,
                                                     self.target_service['shoot_config_resource_type'])
            ammo_resource = self.get_task_resource(self.Context.generate_ammo_task_id,
                                                   self.target_service['ammo_resource_type'])
            self.set_info('Shooting {}...'.format(dc_name))
            self.shoot_via_tank_api(self.target_service['nanny_service_prefix'] + dc_name, config_resource,
                                    ammo_resource)
        with self.memoize_stage['get_shoot_results_' + dc_name]:
            shoot_task = self.get_task(self.Context.last_shoot_task_id)
            log.info('shoot {} lunapark link: {}'.format(dc_name, shoot_task.Parameters.lunapark_link))
            log.info('shoot {} lunapark job id: {}'.format(dc_name, shoot_task.Parameters.lunapark_job_id))
            self.Context.imbalance_rps_list.append(self.get_imbalance_rps(shoot_task))

    def get_imbalance_rps(self, shoot_task):
        lunapark_job_id = shoot_task.Parameters.lunapark_job_id
        if not lunapark_job_id:
            raise TaskFailure('No lunapark job id')
        summary = self.lunaparkClient.get_summary(lunapark_job_id)
        log.info('shoot summary: {}'.format(summary))
        imbalance_rps = summary['imbalance_rps']
        if imbalance_rps == 0 and self.Parameters.start_rps > 0:
            self.set_info('!!WARNING!!: imbalance RPS == 0. Seems that shooting ends before '
                          'the service is unstable.')
            imbalance_rps = int(self.Parameters.target_rps)
        return imbalance_rps

    @staticmethod
    def get_task_resource(task_id, resource_type):
        resource = sdk2.Resource.find(resource_type=resource_type, task_id=task_id).first()
        if not resource:
            raise TaskFailure('resource {} of task {} not found'.format(resource_type, task_id))
        return resource

    @staticmethod
    def get_task(task_id):
        task = sdk2.Task.find(id=task_id, status=ctt.Status.SUCCESS).first()
        if not task:
            raise TaskFailure('Task id={} with status SUCCESS not found'.format(task_id))
        return task

    def calculate_rps(self):
        log.info('starting calculate_rps')
        calculate_rps_task_class = self.target_service['calculate_rps_task']
        if self.target_service['name'] == 'checkouter':
            calculate_rps_task = calculate_rps_task_class(
                self, yql_token_owner=self.Parameters.yql_token_owner,
                yql_token_name=self.Parameters.yql_token_name,
                shoot_only_archive_api=self.Parameters.shoot_only_archive_api,
                use_archive_api_statistics=self.Parameters.use_archive_api_statistics)
        else:
            calculate_rps_task = calculate_rps_task_class(self, yql_token_owner=self.Parameters.yql_token_owner,
                                                          yql_token_name=self.Parameters.yql_token_name)
        calculate_rps_task.enqueue()
        self.Context.calculate_rps_task_id = str(calculate_rps_task.id)
        raise sdk2.WaitTask([calculate_rps_task.id], ctt.Status.Group.SUCCEED, wait_all=True, timeout=3600)

    def generate_ammo(self):
        rps_resource = None
        if not self.Parameters.use_custom_rps_ratio:
            rps_resource = self.get_task_resource(self.Context.calculate_rps_task_id,
                                                  self.target_service['rps_resource_type'])
            log.debug('found rps_resource {}'.format(rps_resource.id))
        if not self.Parameters.autostop_config:
            autostop_config = SERVICES[str(self.Parameters.target_service)]['default_autostop_config']
            log.debug('using default autostop config {}'.format(autostop_config))
        else:
            autostop_config = self.Parameters.autostop_config
        log.info('starting generate_ammo')
        generate_ammo_task = self.target_service['generate_ammo_task'](
            self, imbalance_shooting=self.Parameters.imbalance_shooting,
            autostop_config=autostop_config,
            start_rps=self.Parameters.start_rps,
            target_rps=self.Parameters.target_rps,
            rps_increase_step=self.Parameters.rps_increase_step,
            rps_step_duration_seconds=self.Parameters.rps_step_duration_seconds,
            total_rps=self.Parameters.total_rps,
            test_duration_minutes=self.Parameters.test_duration_minutes,
            endpoints_rps_resource=rps_resource,
            custom_endpoints_rps=self.Parameters.custom_rps_ratio,
            checkouter_url=self.Parameters.checkouter_url,
            report_url=self.Parameters.report_url,
            stocks_url=self.Parameters.stocks_url,
            loyalty_url=self.Parameters.loyalty_url,
            skip_options=self.Parameters.skip_options
        )
        generate_ammo_task.enqueue()
        self.Context.generate_ammo_task_id = str(generate_ammo_task.id)
        raise sdk2.WaitTask([generate_ammo_task.id], ctt.Status.Group.SUCCEED, wait_all=True, timeout=3600)

    def shoot_via_tank_api(self, nanny_service_name, config_resource, ammo_resource):
        log.info('starting shoot_via_tank_api for nanny service {}'.format(nanny_service_name))
        shoot_task = ShootViaTankapi(
            self,
            use_public_tanks=True,
            nanny_service=nanny_service_name,
            config_source='resource',
            config_resource=config_resource,
            ammo_source='resource',
            ammo_resource=ammo_resource
        )
        shoot_task.enqueue()
        log.debug('saving last_shoot_task_id={}'.format(shoot_task.id))
        self.Context.last_shoot_task_id = str(shoot_task.id)
        log.debug('saved last_shoot_task_id={}'.format(self.Context.last_shoot_task_id))
        raise sdk2.WaitTask([shoot_task.id], ctt.Status.Group.SUCCEED, wait_all=True,
                            timeout=3600)

    def push_to_solomon(self, sensor_label, value):
        log.info('starting push_to_solomon')
        push_task = PushToSolomon(
            self,
            project_id=self.target_service['solomon_project_id'],
            service_name=self.target_service['solomon_service_name'],
            cluster_name='stable',
            sensor_label=sensor_label,
            value=float(value)
        )
        push_task.enqueue()
        return push_task.id
