import requests
import json
import logging as log
import xml.etree.ElementTree as ET

REQUEST_FAILED__FORMAT = 'Request {} {} failed: {} {}'

CHECKOUTER_LOAD_URL = 'http://checkouter.load.vs.market.yandex.net:39001'
CHECKOUTER_PROD_URL = 'http://checkouter.market.http.yandex.net:39001'

CARTER_PROD_URL = 'http://carter.vs.market.yandex.net:35803/configuration'

NOTIFIER_PROD_URL = 'http://vla1-4731-vla-market-prod-mark-7ac-23830.gencfg-c.yandex.net:23830'

PUSH_API_LOAD_URL = 'http://push-api.load.vs.market.yandex.net:39002'
PUSH_API_PROD_URL = 'http://push-api.vs.market.yandex.net:39002'

IGNORE_CHECKOUTER_PROPERTIES = ['archivingShopIdExclusions',
                                'archiveWithoutCheckingArchivingConditionEnabled',
                                'archiveAPIEnabled',
                                'enabledArchivedHistoryEventGeneration',
                                'enableArchivedOrderDelete',
                                'enableArchivingBulkInsert',
                                'orderArchivingStartPeriod',
                                'tvmRequired']

IGNORE_CARTER_PROPERTIES = ['tvmRequired']
IGNORE_PUSH_API_PROPERTIES = ['tvmRequired']


class PropertySyncer:
    def __init__(self, task):
        self.add_task_info = task.set_info
        self.endpoint = None

    def sync_all_properties(self, carter_load_url, notifier_load_url):
        """Copy properties from PROD to LOAD"""
        self.sync_checkouter_properties('/properties')
        self.sync_checkouter_properties('/permanent-properties')
        self.sync_carter_properties(carter_load_url, '/properties')
        self.sync_notifier_properties(notifier_load_url, '/properties')
        self.sync_pushapi_properties('/properties')

    def get_json_properties(self, url):
        res = self.get_raw_properties(url)
        return json.loads(res.text)

    def get_raw_properties(self, url):
        res = requests.get(url + self.endpoint)
        if res.status_code != 200:
            raise Exception(
                'Request {} {} failed: {} {}'.format(res.request.method, res.url, res.status_code, res.text))
        log.info('Got properties: {}'.format(res.text))
        return res

    def get_xml_properties(self, url):
        res = self.get_raw_properties(url)
        tree = ET.fromstring(res.text)
        return tree

    def sync_json_properties(self, prod_url, load_url, ignore_properties=None):
        if ignore_properties is None:
            ignore_properties = []
        prod_props = self.get_json_properties(prod_url)
        load_props = self.get_json_properties(load_url)
        for property in prod_props:
            if property in ignore_properties:
                self.add_task_info('Ignoring property {}'.format(property))
                continue
            prod_value = prod_props.get(property)
            load_value = load_props.get(property)
            if prod_value != load_value:
                self.log_property_change(load_value, prod_value, property)
                self.set_json_property(load_url, property, prod_value)

    def log_property_change(self, load_value, prod_value, property):
        msg = '{}: {} ----> {}'.format(property, load_value, prod_value)
        log.info(msg)
        self.add_task_info(msg)

    def sync_checkouter_properties(self, endpoint):
        self.endpoint = endpoint
        log.info('syncing checkouter {}'.format(endpoint))
        self.add_task_info('syncing checkouter properties')
        self.sync_json_properties(CHECKOUTER_PROD_URL, CHECKOUTER_LOAD_URL, IGNORE_CHECKOUTER_PROPERTIES)

    def sync_carter_properties(self, load_url, endpoint):
        self.endpoint = endpoint
        log.info('syncing carter {}'.format(endpoint))
        self.add_task_info('syncing carter properties')
        self.sync_json_properties(CARTER_PROD_URL, load_url, IGNORE_CARTER_PROPERTIES)

    def set_json_property(self, url, name, value):
        res = requests.put(url + self.endpoint + '/' + name, data=json.dumps(value),
                           headers={'Content-Type': 'application/json'})
        if res.status_code != 200:
            self.log_error_response(res)

    def set_raw_property(self, url, name, value):
        res = requests.put(url + self.endpoint + '/' + name, data=value)
        if res.status_code != 200:
            self.log_error_response(res)

    def log_error_response(self, res):
        msg = REQUEST_FAILED__FORMAT.format(res.request.method, res.url, res.status_code, res.text)
        self.add_task_info(msg)
        log.error(msg)

    def sync_xml_properties(self, prod_url, load_url, ignore_properties=None, set_raw_value=False):
        prod_xml = self.get_xml_properties(prod_url)
        load_xml = self.get_xml_properties(load_url)
        if ignore_properties is None:
            ignore_properties = []
        for attr_name in prod_xml.attrib:
            if attr_name in ignore_properties:
                self.add_task_info('Ignoring property {}'.format(attr_name))
                continue
            prod_value = prod_xml.attrib.get(attr_name)
            load_value = load_xml.attrib.get(attr_name)
            if prod_value != load_value:
                self.log_property_change(load_value, prod_value, attr_name)
                xml_data = load_xml.copy()
                xml_data.attrib = {attr_name: prod_value}
                if set_raw_value:
                    self.set_raw_property(load_url, attr_name, prod_value)
                else:
                    self.set_xml_property(load_url, xml_data)

    def sync_notifier_properties(self, load_url, endpoint):
        self.endpoint = endpoint
        log.info('syncing notifier {}'.format(endpoint))
        self.add_task_info('syncing notifier properties')
        self.sync_json_properties(NOTIFIER_PROD_URL, load_url, [])

    def sync_pushapi_properties(self, endpoint):
        self.endpoint = endpoint
        log.info('syncing push-api {}'.format(endpoint))
        self.add_task_info('syncing push-api properties')
        self.sync_json_properties(PUSH_API_PROD_URL, PUSH_API_LOAD_URL, IGNORE_PUSH_API_PROPERTIES)

    def set_xml_property(self, url, xml_data):
        xml_string = ET.tostring(xml_data, encoding='UTF-8', method='xml')
        res = requests.put(url + self.endpoint, data=xml_string,
                           headers={'Content-Type': 'application/xml'})
        if res.status_code != 200:
            self.log_error_response(res)


if __name__ == '__main__':
    log.basicConfig(level=log.DEBUG)

    syncer = PropertySyncer(None)
    syncer.endpoint = '/properties'
    # syncer.set_xm_property(CARTER_LOAD_URL, IGNORE_CARTER_PROPERTIES[0], False)
    # syncer.set_xml_property(PUSH_API_LOAD_URL,
    #                         ET.fromstring(
    #                         '<?xml version="1.0" encoding="UTF-8"?><pushApiProperties tvmRequired="false"/>'))
