import logging as log
import requests
import time
import os
import json


class Pusher(object):
    def __init__(self,
                 solomon_api_url,
                 oauth_token,
                 project_id,
                 service_name,
                 cluster_name):
        self.solomon_api_url = solomon_api_url
        self.oauth_token = oauth_token
        self.project_id = project_id
        self.service_name = service_name
        self.cluster_name = cluster_name

    def push_single_sensor(self, sensor_label, value):
        log.info('Pushing sensor "{}" with value {}'.format(sensor_label, value))
        log.info('Project id "{}", service name "{}", cluster_name "{}"'.format(self.project_id, self.service_name,
                                                                                self.cluster_name))
        body = self.__make_data(sensor_label, value)
        log.debug('data to push: ' + body)
        response = requests.post(
            self.solomon_api_url + '/api/v2/push?project={project}&service={service}&cluster={cluster}'.format(
                project=self.project_id, service=self.service_name, cluster=self.cluster_name
            ), headers=self.__make_headers(), data=body)
        if response.status_code != 200:
            raise Exception('Solomon responded with {} {} {}'.format(response.status_code, response.reason,
                                                                     response.text))

    def __make_headers(self):
        return {
            'Authorization': 'OAuth ' + self.oauth_token,
            'Content-Type': 'application/json'
        }

    @staticmethod
    def __make_data(sensor_label, value):
        return json.dumps({
            "sensors": [{
                "labels": {"sensor": sensor_label},
                "ts": int(time.time()),
                "value": int(value)
            }]
        })


def main():
    log.basicConfig(level=log.DEBUG)

    oauth_token = os.getenv('SOLOMON_OAUTH')

    pusher = Pusher(solomon_api_url='http://solomon.yandex.net',
                    oauth_token=oauth_token,
                    project_id='market-checkout',
                    service_name='market-carter',
                    cluster_name='stable'
                    )
    pusher.push_single_sensor(sensor_label=u'maxRps', value=5)


if __name__ == '__main__':
    main()
