# coding: utf-8

import logging
import os
import json
import shutil
import tarfile
from contextlib import closing

from sandbox import sdk2
from sandbox.sdk2 import yav
from sandbox.sdk2.helpers import subprocess
from sandbox.common.types.task import ReleaseStatus
import sandbox.common.types.client as ctc
from sandbox import common

from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment

from sandbox.projects.market import resources
# from sandbox.projects.market.sre.BuildMarketDatasources import MarketDatasourcesStable, MarketDatasourcesPrestable, MarketDatasourcesTesting
from sandbox.projects.market.contentApi import ReleaseStatusParam, untar


class MarketContentApiBundle(sdk2.Resource):
    """
    Ресурс представляет приложение Market Content API, содержащее набор данных (файловые кеши,
    файлы data-getter и пр.)
    Упаковка в один ресурс - способ синхронизации кода Market Content API и данных для этого кода,
    что также позволяет отказаться от поддержки кодом старого формата данных при его изменении
    """
    releasable = True
    releasers = resources.market_content_api


class MarketContentApiBuildBundle(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Задача, готовит ресурс Market Content API bundle, содержащий приложение Content API,
    а также набор данных (файловые кеши, ресурсы data-getter)
    Данные формирутся кодом приложения, что обеспечивает их консистентность с кодом.
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [SandboxJavaJdkEnvironment('1.8.0')]
        client_tags = ctc.Tag.Group.LINUX
        disk_space = 10 * 1024
        ram = 1024

    class Parameters(sdk2.Task.Parameters):
        """
        Параметры задачи, для хранениия и связи с ресурсами, из которых собирается bundle
        """
        app_resource = sdk2.parameters.Resource(
            'Приложение market-content-api',
            resource_type=resources.MARKET_CONTENT_API_APP,
            required=True
        )
        datagetter_resource = sdk2.parameters.Resource(
            'Ресурсы data-getter для market-content-api',
            resource_type=resources.MARKET_DATA_CONTENT_API,
            required=True
        )
        datasources_resource = sdk2.parameters.Resource(
            'Market Datasources',
            # FIXME: invalid argument (SANDBOX-6404)
            # types=[MarketDatasourcesTesting, MarketDatasourcesPrestable, MarketDatasourcesStable],
            required=True
        )
        release_status = ReleaseStatusParam(
            'Тип релиза',
            required=True
        )

        with sdk2.parameters.Output:
            app_description = sdk2.parameters.String(
                'Описание версии приложения'
            )

    def mark_released_resources(self, status, ttl='inf'):
        assert self.parent is not None, 'Parent task is required'
        ttl = self.parent.Parameters.release_ttl
        super(MarketContentApiBuildBundle, self).mark_released_resources(status, ttl)

    def on_execute(self):
        assert self.parent is not None, 'Parent task is required'
        release_status = self.Parameters.release_status
        app_resource = self.Parameters.app_resource
        datagetter_resource = self.Parameters.datagetter_resource
        datasources_resource = self.Parameters.datasources_resource
        logging.info(
            'Release status={}, Input Resources: app.id={}, datagetter.id={}, datasources.id={}'
                .format(release_status, app_resource.id, datagetter_resource.id, datasources_resource.id)
        )
        app_resource_path = sdk2.ResourceData(app_resource).path
        datagetter_resource_path = sdk2.ResourceData(datagetter_resource).path
        datasources_resource_path = sdk2.ResourceData(datasources_resource).path
        logging.debug('Input resources paths: app={}, datagetter={}, datasources={}',
                      app_resource_path, datagetter_resource_path, datasources_resource_path)

        # Готовим файловое окружение
        # Распаковываем входные ресурсы
        # Директория приложения
        app_path = self.path(untar(str(app_resource_path), "."))
        lib_path = app_path.joinpath('lib')
        logging.debug('app_path is: "{}", lib_path is: "{}"'.format(app_path, lib_path))
        assert lib_path.exists()

        external_path = app_path.joinpath('external')

        # Папка куда складываем файлы datasources
        properties_d_path = app_path.joinpath('properties.d')
        properties_d_path.mkdir(exist_ok=True)
        properties_d_path_str = str(properties_d_path)
        untar(str(datasources_resource_path), str(app_path), 1)

        # Папка с данными
        data_path = app_path.joinpath('data')
        data_path.mkdir(exist_ok=True)
        data_getter_path = data_path.joinpath(datagetter_resource_path.name)
        shutil.copytree(str(datagetter_resource_path), str(data_getter_path))
        # Папка данных, куда будем сохранять файловые кеши
        cache_path = data_path.joinpath('cache')
        cache_path.mkdir(exist_ok=True)
        # conf папка - подключаем в classpath
        conf_path = app_path.joinpath('conf')

        # vault secrets
        vaults_path = self.path('vaults')
        vaults_path.mkdir(exist_ok=True)
        # Создаем файл для TVM
        tvm_data = sdk2.Vault.data(self.parent.Parameters.tvm_stable_vault_name
            if release_status in [ReleaseStatus.STABLE, ReleaseStatus.PRESTABLE]
            else self.parent.Parameters.tvm_tesing_vault_name
        )
        tvm_path = vaults_path.joinpath('market_corba-tvm.json')
        tvm_path.write_text(unicode(tvm_data))
        # billing.properties
        if release_status in [ReleaseStatus.STABLE, ReleaseStatus.PRESTABLE]:
            billing_data = sdk2.Vault.data(self.parent.Parameters.billing_props_vault_name)
            billing_path = vaults_path.joinpath('billing.properties')
            billing_path.write_text(unicode(billing_data))

        # datasources.properties
        if release_status in [ReleaseStatus.STABLE, ReleaseStatus.PRESTABLE]:
            secret = yav.Secret(self.parent.Parameters.pgaas_connection_production_secret_id)
        else:
            secret = yav.Secret(self.parent.Parameters.pgaas_connection_testing_secret_id)

        datasources_data = secret.data()["datasources.properties"]
        datasources_path = vaults_path.joinpath('datasources.properties')
        datasources_path.write_text(unicode(datasources_data))

        logging.info('Files prepared: app.path: {}, vaults.path={}', app_path, vaults_path)

        # Добавляем в описание задачи информацию о ресурсе и таске
        app_info_parts = []
        app_info_parts.append('Resource: https://sandbox.yandex-team.ru/resource/{}/view'.format(str(app_resource.id)))
        app_info_parts.append('Task: https://sandbox.yandex-team.ru/task/{}/view'.format(str(app_resource.task.id)))

        data_getter_info_parts = []
        data_getter_info_parts.append('Resource: https://sandbox.yandex-team.ru/resource/{}/view'.format(str(datagetter_resource.id)))
        data_getter_info_parts.append('Task: https://sandbox.yandex-team.ru/task/{}/view'.format(str(datagetter_resource.task.id)))

        self.Parameters.app_description = 'Content API information:\n   {}\n\nData Getter information:\n  {}'.format('\n  '.join(app_info_parts), '\n  '.join(data_getter_info_parts))

        tvm_secret = sdk2.Vault.data(self.parent.Parameters.tvm_stable_vault_name_2
            if release_status in [ReleaseStatus.STABLE, ReleaseStatus.PRESTABLE]
            else self.parent.Parameters.tvm_testing_vault_name_2
        )

        # Переменные окружения для запуска генератора кеш-файлов
        penv = os.environ.copy()
        penv['PROPERTIES_DIR'] = properties_d_path_str
        penv['TVM_SECRET'] = tvm_secret
        # System Property 'environment', от которой зависят адреса внешних источников данных
        app_environment = 'production' if release_status == ReleaseStatus.STABLE else release_status

        with sdk2.helpers.ProcessLog(self, logger='cache-dump') as pl:
            pl.logger.propagate = 1
            retcode = subprocess.Popen(
                'export LD_LIBRARY_PATH=$LD_LIBRARY_PATH:{}; java -cp "{}:{}:{}:{}:{}" -Djava.net.preferIPv6Addresses=true -Denvironment={} -Dcache.dir={}'
                ' -Dmarket_data_getter.base={} -Doracle.net.tns_admin={} ru.yandex.market.api.CachesDumper'
                    .format(
                        str(external_path),
                        str(lib_path.joinpath('*')),
                        properties_d_path_str,
                        str(conf_path),
                        str(external_path),
                        str(vaults_path),
                        app_environment,
                        str(cache_path),
                        str(data_getter_path),
                        properties_d_path_str
                    ),
                shell=True, stdout=pl.stdout, stderr=subprocess.STDOUT, cwd=str(app_path), env=penv
            ).wait()
        # явно удаляем, согласно sox
        shutil.rmtree(path=str(vaults_path), ignore_errors=True)
        if retcode:
            raise common.errors.SubprocessError('Java process exited with non-zero return code {}'.format(retcode))

        bundle_path = str(app_path) + '.tar.gz'
        logging.info('Creating resource %s in path: %s', MarketContentApiBundle.type, bundle_path)
        bundle = MarketContentApiBundle(
            self,
            'Yandex Market Content API bundle (application + data)\n {}'.format(self.Parameters.app_description),
            bundle_path
        )
        data = sdk2.ResourceData(bundle)

        # Удаляем приложение и конфиги, которые приходят в RTC отдельно
        shutil.rmtree(path=str(app_path.joinpath('bin')), ignore_errors=True)
        shutil.rmtree(path=str(app_path.joinpath('conf')), ignore_errors=True)
        shutil.rmtree(path=str(app_path.joinpath('external')), ignore_errors=True)
        shutil.rmtree(path=str(app_path.joinpath('lib')), ignore_errors=True)
        shutil.rmtree(path=str(app_path.joinpath('properties.d')), ignore_errors=True)

        # Нужно исследовать, возможно data-getter данные для кешей и не нужны
        # shutil.rmtree(path=str(data_getter_path), ignore_errors=True)

        logging.info('Creating bundle tarball %s', bundle_path)
        with closing(tarfile.open(name=bundle_path, mode='w:gz')) as tar:
            tar.add(str(app_path), arcname=app_path.name)

        logging.info('Marking resource as ready')
        data.ready()
