# coding: utf-8

import logging
from datetime import datetime, timedelta
from sandbox import sdk2
from sandbox.sdk2 import yav
import sandbox.common.utils as util
import sandbox.common.types.task as ctt
import sandbox.common.types.notification as ctn
from sandbox.common.types.task import ReleaseStatus

from sandbox.projects.market import resources
from sandbox.projects.market.sre.BuildMarketDatasources import MarketDatasourcesStable, MarketDatasourcesTesting
from sandbox.projects.market.contentApi.MarketContentApiBuildBundle import MarketContentApiBuildBundle, MarketContentApiBundle
from sandbox.projects.market.contentApi import RELEASE_STATUSES


class MarketContentApiReleaseBundles(sdk2.Task):
    """
    Задача предназначена для формирования и релиза ресурсов Market Content API bundle,
    содержащих приложение Content API, а также набор данных (файловые кеши, ресурсы data-getter).
    Релизы формируются в статусах 'testing', 'prestable' и 'stable'.
    Bundle ресурса формируется из релизов ресурсов 'MARKET_CONTENT_API_APP' (Приложение API),
    'MARKET_DATA_CONTENT_API' (данные data-getter), 'MARKET_DATASOURCES-*' (настройки datasouces) дочерней
    задачей, при успешном завершении которой, ресурс релизится в одноименном статусе.
    Новый bundle релиз формируется при появлении новых релизов ресурсов приложения и данных data-getter в
    соответсвующих статусах. Если новых релизов указанных ресурсов не появляется, то новый bundle релиз
    формируется с интервалом указанным в параметре задачи, это необходимо для поддержки обновления данных
    во внешних источниках.
    Задача должна периодически запускаться scheduller'ом
    """

    class Parameters(sdk2.Task.Parameters):
        data_update_interval = sdk2.parameters.Integer(
            'Интервал времени (мин.) периодического обновления релизов'
            ' (для поддержки изменений данных во внешних источниках)',
            required=True,
            default_value=60
        )

        release_ttl = sdk2.parameters.Integer(
            'Время жизни релизов, дни',
            required=True,
            default_value=30
        )

        with sdk2.parameters.CheckGroup(
            'Периодическое обновление релизов для:',
            default=[ReleaseStatus.TESTING]
        ) as update_flags:
            update_flags.choices = [(_, _) for _ in RELEASE_STATUSES]

        tvm_tesing_vault_name = sdk2.parameters.String(
            'TVM secret (testing) vault name',
            default_value='market_corba_test_dev_tvm',
            required=True
        )

        tvm_stable_vault_name = sdk2.parameters.String(
            'TVM secret (stable, prestable) vault name',
            default_value='market_corba_stable_tvm',
            required=True
        )

        tvm_testing_vault_name_2 = sdk2.parameters.String(
            'TVM secret (testing) vault name new format',
            default_value='market_corba_test_tvm',
            required=True
        )

        tvm_stable_vault_name_2 = sdk2.parameters.String(
            'TVM secret (stable, prestable) vault name new format',
            default_value='market_corba_prod_tvm',
            required=True
        )

        billing_props_vault_name = sdk2.parameters.String(
            'billing.properties (stable, prestable) vault name',
            default_value='market_content_api_sox_properties',
            required=True
        )

        error_email = sdk2.parameters.String(
            'Users for tasks error message',
            default_value='dimkarp93',
            required=True
        )

        pgaas_connection_testing_secret_id = sdk2.parameters.String(
            'Connection properties for PGaaS (testing)',
            default_value='sec-01f4h6yy8mfzestzkyst4pm51m',
            required=True
        )

        pgaas_connection_production_secret_id = sdk2.parameters.String(
            'Connection properties for PGaaS (production)',
            default_value='sec-01f5myj7mhg7yfyb79qxd71x48',
            required=True
        )

    def _check_and_create_subtask(self, release_status):
        """
        Логика проверки необходимости формирования нового ресурса:
        1. При появлении новых ресурсов приложения или данных data-pusher
        2. Для поддержки обновления данных во внешних источниках, с периодом self.Parameters.data_update_interval
        :param release_status: статус релиза, для которого делаем проверку
        :return: новый subtask - если необходимо формировать новый bundle, иначе - None
        """
        def _find_last_release(resourse_type, release_status):
            res = sdk2.Resource.find(resourse_type, attrs=dict(released=release_status)).order(-sdk2.Resource.id).first()
            if not res:
                logging.warning("Last release of type '{}' with release_status='{}' not found"
                    .format(str(resourse_type), release_status))
            return res

        logging.info('Checking for {}'.format(release_status))

        old_bundle = _find_last_release(MarketContentApiBundle, release_status)
        old_app_release = old_bundle.task.Parameters.app_resource if old_bundle else None

        last_app_release = _find_last_release(resources.MARKET_CONTENT_API_APP, release_status)
        # Предыдущий APP ресурс может быть переведен из testing'а в prestable, etc.
        if last_app_release is None or (old_app_release and old_app_release.id > last_app_release.id):
            last_app_release = old_app_release

        last_dg_release = _find_last_release(resources.MARKET_DATA_CONTENT_API,
                        ReleaseStatus.STABLE if release_status == ReleaseStatus.PRESTABLE else release_status)
        ds_release_type = {
            ReleaseStatus.STABLE: MarketDatasourcesStable,
            ReleaseStatus.PRESTABLE: MarketDatasourcesStable,
            ReleaseStatus.TESTING: MarketDatasourcesTesting
        }.get(release_status)
        last_ds_release = _find_last_release(ds_release_type, ReleaseStatus.STABLE)
        if not last_app_release or not last_dg_release or not last_ds_release:
            return None

        def create_subtask(release_cause):
            subtask = MarketContentApiBuildBundle(
                self,
                description=release_cause,
                app_resource=last_app_release,
                datagetter_resource=last_dg_release,
                datasources_resource=last_ds_release,
                release_status=release_status
            )
            return subtask

        if not old_bundle:
            return create_subtask("Не найден предыдущий bundle ({})".format(release_status))

        logging.info('Old bundle id={}, created={}'.format(old_bundle.id, old_bundle.created))
        # Проверяем параметры
        old_task = old_bundle.task
        app_old_id = old_task.Parameters.app_resource.id
        dg_old_id = old_task.Parameters.datagetter_resource.id
        app_diff = app_old_id != last_app_release.id
        dg_diff = dg_old_id != last_dg_release.id
        if dg_diff:
            idslog = lambda x, y: str(x) if x == y else (str(x) + "->" + str(y))
            logging.info(
                'Last releases (bundled->newly): app.id={}, data-getter.id={}; New bundle to be created...',
                idslog(app_old_id, last_app_release.id), idslog(dg_old_id, last_dg_release.id)
            )
            changes = []
            r_str = lambda r: str(r.type) + '.' + str(r.id)
            if app_diff: changes.append(r_str(last_app_release))
            if dg_diff: changes.append(r_str(last_dg_release))
            return create_subtask("Обнаружены новые релизы ({}): {}".format(release_status, ', '.join(changes)))
        if release_status in self.Parameters.update_flags:
            delta = datetime.now(old_bundle.created.tzinfo) - old_bundle.created
            if delta > timedelta(minutes=self.Parameters.data_update_interval):
                logging.info('Old bundle too old on {}'.format(delta))
                return create_subtask("Предыдущий bundle слишком старый ({})".format(release_status))
        return None

    def _create_release(self, task_id, status=ReleaseStatus.CANCELLED, subject=None, comments=None,
        addresses_to=None, addresses_cc=None, additional_parameters=None):
        """
        Release task with given id.

        :param task_id: id of task to release
        :param status: release status (cancelled, stable, prestable, testing, unstable)
        :param subject: release notification subject
        :param comments: release notification body
        :param addresses_to: list of recipient logins
        :param addresses_cc: list of CC-recipient logins
        :param additional_parameters: dictionary with params to pass to `on_release` method
        :return: release id or None if task releasing failed
        :rtype: int
        """
        task_id = int(task_id)
        additional_parameters = additional_parameters or {}
        additional_parameters.update(author=self.author)
        subject = subject or ''
        comments = comments or ''
        addresses_to = addresses_to or []
        addresses_cc = addresses_cc or []

        return self.server.release({
            'to': addresses_to,
            'params': additional_parameters,
            'task_id': task_id,
            'cc': addresses_cc,
            'message': comments,
            'type': status,
            'subject': subject,
        })

    def _send_error_email(self, tasks):
        mail_subj = 'Unsuccessful release Content API Bundle tasks'
        mail_body = 'Unsuccessful sandbox task:\n' + \
                    '\n'.join([util.get_task_link(task.id) for task in tasks]) + \
                    '\n--\nsandbox \n https://sandbox.yandex-team.ru/ \n'
        try:
            self.server.notification(
                subject=mail_subj,
                body=mail_body,
                recipients=str(self.Parameters.error_email).split(),
                transport=ctn.Transport.EMAIL,
                type=ctn.Type.TEXT,
                charset=ctn.Charset.UTF,
                task_id=self.id,
                view=ctn.View.DEFAULT
            )
        except Exception:
            logging.exception('')

    def on_execute(self):
        with self.memoize_stage.first_step:
            subtasks = []
            for release_status in RELEASE_STATUSES:
                sub_task = self._check_and_create_subtask(release_status)
                if sub_task:
                    subtasks.append(sub_task)
                    sub_task.enqueue()
            if len(subtasks) > 0:
                raise sdk2.WaitTask(
                    subtasks,
                    ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                    wait_all=True,
                    timeout=1800
                )
            else:
                self.set_info('Не было создано новых релизов')
        with self.memoize_stage.second_step:
            subtasks = self.find(MarketContentApiBuildBundle)
            unsuccess = []
            for subtask in subtasks:
                if subtask.status == ctt.Status.SUCCESS:
                    resp = self._create_release(
                        task_id=subtask.id,
                        status=subtask.Parameters.release_status,
                        subject=subtask.Parameters.description,
                        comments=subtask.Parameters.app_description
                    )
                    logging.info('Released task.id={}'.format(subtask.id, resp))
                    self.set_info(subtask.Parameters.description)
                else:
                    unsuccess.append(subtask)
            if len(unsuccess) > 0:
                self._send_error_email(unsuccess)
