from datetime import datetime, timedelta

import pandas as pd

import sandbox.projects.market.contentApi.lib.solomon_client as slmn

CHECKOUTER_SERVICE_NAME = 'market-checkouter'
CHECKOUTER_ORDERS_FORECAST_SENSOR = 'ordersForecast'

AF_SERVICE_NAME = 'antifraud_orders'
AF_RPS_CAPACITY_SENSOR = 'capacity_rps_DC-1'
AF_ORDERS_CAPACITY_SENSOR = 'capacity_orders_DC-1'
AF_DAU_CAPACITY_SENSOR = 'capacity_dau_DC-1'
AF_ORDERS_FORECAST_SENSOR = 'forecast_orders'
AF_DAU_FORECAST_SENSOR = 'forecast_dau'

DAILY_ORDERS_COUNT_SENSOR = 'dailyOrdersCount'
CARTER_DAU_FORECAST_SENSOR = 'dauForecast'


def ts_to_start_of_day(timestamp):
    return datetime.timestamp(datetime.combine(datetime.fromtimestamp(timestamp), datetime.min.time()))


def solomon_checkouter(token):
    return slmn.SolomonClient(
        oauth_token=token,
        project_id='market-checkout',
        cluster_name='stable',
        solomon_api_url='http://solomon.yandex.net')


def solomon_af(token):
    return slmn.SolomonClient(
        oauth_token=token,
        project_id='market-content-api',
        cluster_name='production',
        solomon_api_url='http://solomon.yandex.net')


def get_orders_forecast(solomon, date_from):
    o_forecast_ts, order_forecast_values = solomon.get_values(
        date_from=date_from,
        service_name=CHECKOUTER_SERVICE_NAME,
        sensor_name=CHECKOUTER_ORDERS_FORECAST_SENSOR)
    if len(o_forecast_ts) == 0:
        raise Exception('Cannot calculate reservation: no data in "{}"'.format(CHECKOUTER_ORDERS_FORECAST_SENSOR))
    return order_forecast_values[len(order_forecast_values) - 1]


def get_dau_forecast(solomon, date_from):
    dau_forecast_ts, dau_forecast_values = solomon.get_values(
        date_from=date_from,
        service_name=CHECKOUTER_SERVICE_NAME,
        sensor_name=CARTER_DAU_FORECAST_SENSOR)
    if len(dau_forecast_ts) == 0:
        raise Exception('Cannot calculate reservation: no data in "{}"'.format(CARTER_DAU_FORECAST_SENSOR))
    return dau_forecast_values[len(dau_forecast_values) - 1]


def get_conversions(solomon):
    forecast_date_from = datetime.now() - timedelta(days=30)
    o_forecast_ts, order_forecast_values = solomon.get_values(
        date_from=forecast_date_from,
        service_name=CHECKOUTER_SERVICE_NAME,
        sensor_name=CHECKOUTER_ORDERS_FORECAST_SENSOR)
    if len(o_forecast_ts) == 0:
        raise Exception('Cannot calculate reservation: no data in "{}"'.format(CHECKOUTER_ORDERS_FORECAST_SENSOR))
    measures = {}
    for i in range(len(o_forecast_ts)):
        ts = ts_to_start_of_day(o_forecast_ts[i])
        measures[ts] = {}
        measures[ts]['orders'] = order_forecast_values[i]

    dau_forecast_ts, dau_forecast_values = solomon.get_values(
        date_from=forecast_date_from,
        service_name=CHECKOUTER_SERVICE_NAME,
        sensor_name=CARTER_DAU_FORECAST_SENSOR)
    if len(dau_forecast_ts) == 0:
        raise Exception('Cannot calculate reservation: no data in "{}"'.format(CARTER_DAU_FORECAST_SENSOR))
    for i in range(len(dau_forecast_ts)):
        ts = ts_to_start_of_day(dau_forecast_ts[i])
        measures[ts] = {}
        measures[ts]['dau'] = dau_forecast_values[i]

    conversions = {}
    for ts, measure in measures:
        dau = measure['dau']
        orders = measure['orders']
        if dau is None or orders is None:
            continue
        conversions[ts] = orders / dau
    return conversions


def get_rps_quantile(solomon, date_from, date_to, quantile):
    rps_ts, rps_values = solomon.get_values(
        date_from=date_from,
        date_to=date_to,
        service_name=AF_SERVICE_NAME,
        sensor_name='RPS',
        other_keys={'environment': 'PRODUCTION', 'host': '-'})
    series = pd.DataFrame(rps_values).quantile(quantile)
    return pd.to_numeric(series)[0]


def get_orders_count(solomon, date_from, date_to):
    if date_to is None:
        date_to = datetime.now()
    o_ts, o_values = solomon.get_values(
        date_from=date_from,
        date_to=date_to,
        service_name=CHECKOUTER_SERVICE_NAME,
        sensor_name=DAILY_ORDERS_COUNT_SENSOR)
    return sum(o_values)


def get_af_last_shoot_result(solomon, date_from):
    ts, values = solomon.get_values(
        date_from=date_from,
        service_name=AF_SERVICE_NAME,
        sensor_name=AF_RPS_CAPACITY_SENSOR)
    if len(values) == 0:
        return None
    return values[len(values) - 1]


def push_orders_capacity(solomon, capacity, ts):
    solomon.push_sensor(
        service_name=AF_SERVICE_NAME,
        sensor_name=AF_ORDERS_CAPACITY_SENSOR,
        ts=datetime.timestamp(ts),
        value=capacity
    )


def push_dau_capacity(solomon, capacity, ts):
    solomon.push_sensor(
        service_name=AF_SERVICE_NAME,
        sensor_name=AF_DAU_CAPACITY_SENSOR,
        ts=datetime.timestamp(ts),
        value=capacity
    )


def push_orders_forecast(solomon, forecast, ts):
    solomon.push_sensor(
        service_name=AF_SERVICE_NAME,
        sensor_name=AF_ORDERS_FORECAST_SENSOR,
        ts=datetime.timestamp(ts),
        value=forecast
    )


def push_dau_forecast(solomon, forecast, ts):
    solomon.push_sensor(
        service_name=AF_SERVICE_NAME,
        sensor_name=AF_DAU_FORECAST_SENSOR,
        ts=datetime.timestamp(ts),
        value=forecast
    )
