import json
import logging as log
import math
import time
from datetime import datetime, timedelta
from collections import Counter

import requests

INTERVAL_DAYS = 7


class SolomonClient(object):
    def __init__(self,
                 solomon_api_url,
                 oauth_token,
                 project_id,
                 cluster_name):
        self.solomon_api_url = solomon_api_url
        self.oauth_token = oauth_token
        self.project_id = project_id
        self.cluster_name = cluster_name

    def push_sensor(self, service_name, sensor_name, value, ts):
        log.info('Pushing sensor "{}" with value {}'.format(sensor_name, value))
        log.info('Project id "{}", service name "{}", cluster_name "{}"'.format(self.project_id, service_name,
                                                                                self.cluster_name))
        if ts is None:
            ts = int(time.time())
        body = self.__make_data(sensor_name, value, ts)
        log.debug('data to push: ' + body)
        response = requests.post(
            self.solomon_api_url + '/api/v2/push?project={project}&service={service}&cluster={cluster}'.format(
                project=self.project_id, service=service_name, cluster=self.cluster_name
            ), headers=self.__make_headers(), data=body)
        if response.status_code != 200:
            raise Exception('Solomon responded with {} {} {}'.format(response.status_code, response.reason,
                                                                     response.text))

    def get_values(self, date_from, service_name, sensor_name, smooth=False, date_to=None,
                   other_keys={}):
        if date_to is None:
            date_to = datetime.now()

        if not smooth:
            result = [[], []]
            intervals = self.split_intervals(date_from, date_to, INTERVAL_DAYS)
            for intv in intervals:
                start = intv[0]
                end = intv[1]
                log.info('Getting data for service: {} sensor: {} interval: {} - {}'
                         .format(service_name, sensor_name, start, end))
                ts, vals = self.__get_values(date_from=start,
                                             date_to=end,
                                             service_name=service_name,
                                             sensor_name=sensor_name,
                                             smooth=smooth,
                                             other_keys=other_keys)
                result[0] += ts
                result[1] += vals
            return result
        else:
            return self.__get_values(
                date_from=date_from,
                date_to=date_to,
                service_name=service_name,
                sensor_name=sensor_name,
                other_keys=other_keys,
                smooth=smooth)

    def __get_values(self, date_from, date_to, service_name, sensor_name, other_keys, smooth=False):
        headers = {
            'Content-Type': 'application/json',
            'Authorization': 'OAuth {}'.format(self.oauth_token),
            'Accept': 'application/json'
        }
        keys = {'cluster': self.cluster_name, 'service': service_name, 'sensor': sensor_name}
        for k, v in other_keys.items():
            keys[k] = v
        programs = []
        for k, v in keys.items():
            programs.append(str(k) + '=\"' + str(v) + '\"')
        program = '{' + ', '.join(programs) + '}'
        if smooth:
            program = 'moving_avg({}, 5d)'.format(program)
        url = self.solomon_api_url + '/api/v2/projects/' + self.project_id + '/sensors/data'
        data = {
            "downsampling": {
                "disabled": not smooth
            },
            "forceCluster": "",
            "from": self.__format_datetime(date_from),
            "program": program,
            "to": self.__format_datetime(date_to),
        }

        log.debug('solomon get values data: {}'.format(data))
        res = requests.post(url, data=json.dumps(data), headers=headers)
        if res.status_code != 200:
            raise Exception(
                'Failed to get data. Solomon responded: {} {}'.format(res.status_code, res.text))
        log.debug('Solomon values response: {}'.format(res.text))
        response_data = json.loads(res.text)
        try:
            vector = response_data['vector']
            counter = Counter({})
            timestamps = set()
            for item in vector:
                timeseries = item['timeseries']
                if timeseries is None or 'values' not in timeseries or 'timestamps' not in timeseries:
                    continue
                values = timeseries['values']
                timestamps = list(map(lambda ts: ts / 1000, timeseries['timestamps']))
                cur_dict = {}
                for i in range(len(timestamps)):
                    ts = timestamps[i]
                    val = values[i]
                    cur_dict[ts] = val
                    timestamps.append(ts)
                counter = counter + Counter(cur_dict)
            timestamps = sorted(timestamps)
            values = []
            for t in timestamps:
                values.append(counter.get(t))
            return timestamps, values
        except (IndexError, KeyError):
            log.exception('Error getting values:')
            return [], []

    @staticmethod
    def __format_datetime(dt):
        return dt.strftime('%Y-%m-%dT%H:%M:%S.%f')[:-3] + 'Z'

    @staticmethod
    def split_intervals(date_from, date_to, interval_days):
        log.debug('split_intervals')
        log.debug('date_from: {}'.format(date_from))
        log.debug('date_to: {}'.format(date_to))
        intervals_count = \
            ((date_to - date_from).total_seconds() / interval_days) / timedelta(days=1).total_seconds()
        log.debug('intervals_count: ' + str(intervals_count))
        previous_date_to = date_from
        intervals = []
        for i in range(1, math.ceil(intervals_count)):
            new_date_to = date_from + timedelta(days=interval_days) * i
            intervals.append([previous_date_to, new_date_to])
            previous_date_to = new_date_to
        intervals.append([previous_date_to, date_to])
        return intervals

    def __make_headers(self):
        return {
            'Authorization': 'OAuth ' + self.oauth_token,
            'Content-Type': 'application/json'
        }

    @staticmethod
    def __make_data(sensor_label, value, ts):
        return json.dumps({
            "sensors": [{
                "labels": {"sensor": sensor_label},
                "ts": ts,
                "value": value
            }]
        })
