# -*- coding: utf-8 -*-

import logging
import tarfile
import os
import sandbox.sdk2.path as spath

from sandbox.projects.common import binary_task
from sandbox import sdk2
from datetime import datetime

from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common.ya_deploy import release_integration
from sandbox.projects.market.content_storage.CsDataCreator.lib import yt2csv


class MarketCsDataTest(sdk2.Resource):
    """ Market content-storage data from mbo (testing) """
    releasable = True
    auto_backup = True
    ttl = 14
    last_mbo_export = sdk2.resource.Attributes.String("Last used path of mbo/export/recent yt-table")


class MarketCsDataProd(sdk2.Resource):
    """ Market content-storage data from mbo (production) """
    releasable = True
    auto_backup = True
    ttl = 14
    last_mbo_export = sdk2.resource.Attributes.String("Last used path of mbo/export/recent yt-table")


class MarketCsPersDataTest(sdk2.Resource):
    """ Market content-storage data from pers (testing) """
    releasable = True
    auto_backup = True
    ttl = 14
    last_pers_path = sdk2.resource.Attributes.String("Last used path of pers-grade/tables/model_rating/current yt-table")


class MarketCsPersDataProd(sdk2.Resource):
    """ Market content-storage data from pers (testing) """
    releasable = True
    auto_backup = True
    ttl = 14
    last_pers_path = sdk2.resource.Attributes.String("Last used path of pers-grade/tables/model_rating/current yt-table")


class MarketCsRecomData(sdk2.Resource):
    """ Market content-storage data from recommendations (testing + prod) """
    releasable = True
    auto_backup = True
    ttl = 14
    last_recom_rb_path = sdk2.resource.Attributes.String("Last used path of reasons_ro_buy yt-file")


class MarketCsPostgreData(sdk2.Resource):
    """ Market content-storage data to init postgre client """
    releasable = True
    auto_backup = True
    ttl = 14


class MarketCsDataCreator(release_integration.ReleaseToYaDeployTask2, binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        description = "Create data for content-storage"

        is_testing = sdk2.parameters.Bool("Load testing data", required=True)
        check_last_resource = sdk2.parameters.Bool("Check relevance of last resource",  default=True, required=True)
        need_create_mbo_data = sdk2.parameters.Bool("Need create mbo data resource",  default=True, required=False)
        need_create_pers_data = sdk2.parameters.Bool("Need create pers data resource",  default=True, required=False)
        need_create_recom_data = sdk2.parameters.Bool("Need create reasons_to_buy data resource",  default=True, required=False)

        yt_token = sdk2.parameters.String("YT Token Secret", default='sec-01fnp1gcd0hsfh2jvcksev3nkr', required=True)
        yp_token = sdk2.parameters.String("YP Token Secret", default='sec-01fy3w8q2q0y41hztt8vqf56p8', required=True)
        yt_cluster = sdk2.parameters.String("Yt cluster", required=True)

        vendors_table_path = sdk2.parameters.String("Vendors table path", required=False)
        nids_table_path = sdk2.parameters.String("Nids table path", required=False)
        hids_table_path = sdk2.parameters.String("Hids table path", required=False)
        params_table_path = sdk2.parameters.String("Params table path", required=False)
        gumoful_templates_table_path = sdk2.parameters.String("Gumofuls table path", required=False)

        pers_table_path = sdk2.parameters.String("Pers table path", required=False)
        reasons_to_buy_path = sdk2.parameters.String("Recom reasons_to_buy yt path", required=False)
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Output():
            has_data_prepared = sdk2.parameters.Bool("Task has a data prepared", default=False)
            task_status = sdk2.parameters.String("Task status", default='Unknown error')

        def all_mbo_tables_defined(self):
            for table in [
                self.vendors_table_path,
                self.nids_table_path,
                self.hids_table_path,
                self.params_table_path,
                self.gumoful_templates_table_path
            ]:
                if table is None or len(table) == 0:
                    return False

            return True

        def pers_table_defined(self):
            return self.pers_table_path is not None and len(self.pers_table_path) > 0

        def recom_table_defined(self):
            return self.reasons_to_buy_path is not None and len(self.reasons_to_buy_path) > 0

    def get_yp_oauth_token(self):
        yp_secret = sdk2.yav.Secret(self.Parameters.yp_token)
        return yp_secret.data()['yp_token']

    def _get_yt_client(self):
        import yt.wrapper as yt

        logging.info('Create yt client')
        yt_secret = sdk2.yav.Secret(self.Parameters.yt_token)
        token = yt_secret.data()['yt_cs_testing']

        return yt.YtClient(
            proxy=self.Parameters.yt_cluster,
            token=token
        )

    def _get_real_path(self, yt_client, table_path, to_dir=False):
        from yt.wrapper.ypath import ypath_dirname
        link_path = ''
        if to_dir:
            link_path = ypath_dirname(table_path)
        else:
            link_path = table_path
        return yt_client.get(link_path+"/@path")  # получаем путь, на который указывает ссылка

    def on_execute(self):
        logging.info('Start a CsDataCreator.')

        yt_client = self._get_yt_client()

        need_create_mbo_data = self.Parameters.need_create_mbo_data
        need_create_pers_data = self.Parameters.need_create_pers_data
        need_create_recom_data = self.Parameters.need_create_recom_data

        if need_create_mbo_data and not self.Parameters.all_mbo_tables_defined():
            raise SandboxTaskFailureError('The task should create mbo data, but not all mbo tables are defined.')

        if need_create_pers_data and not self.Parameters.pers_table_defined():
            raise SandboxTaskFailureError('The task should create pers data, but pers table is not defined.')

        if need_create_recom_data and not self.Parameters.recom_table_defined():
            raise SandboxTaskFailureError('The task should create recom data, but recom table is not defined.')

        mbo_export_path = ''
        pers_path = ''
        reasons_to_buy_path = ''
        if self.Parameters.check_last_resource:
            logging.info('Start checking the last resource.')
            last_resource_mbo = None
            if need_create_mbo_data:
                mbo_export_path = self._get_real_path(yt_client, self.Parameters.vendors_table_path, True)
                logging.info('Actual mbo_export_path={}'.format(mbo_export_path))
                if self.Parameters.is_testing:
                    last_resource_mbo = MarketCsDataTest.find(
                        attrs={"released": "stable"}
                    ).first()
                else:
                    last_resource_mbo = MarketCsDataProd.find(
                        attrs={"released": "stable"}
                    ).first()

                if last_resource_mbo is not None:
                    logging.info('The last resource is detected. Id={}'.format(last_resource_mbo.id))
                    logging.info('Last mbo_export={}'.format(last_resource_mbo.last_mbo_export))

                    if last_resource_mbo.last_mbo_export == mbo_export_path:
                        need_create_mbo_data = False
                    else:
                        logging.info('The last mbo resource contains outdated data.')
                else:
                    logging.info('Any relevant mbo resource is not found.')

            last_resource_pers = None
            if need_create_pers_data:
                pers_path = self._get_real_path(yt_client, self.Parameters.pers_table_path)
                logging.info('Actual pers_path={}'.format(pers_path))
                if self.Parameters.is_testing:
                    last_resource_pers = MarketCsPersDataTest.find(
                        attrs={"released": "stable"}
                    ).first()
                else:
                    last_resource_pers = MarketCsPersDataProd.find(
                        attrs={"released": "stable"}
                    ).first()

                if last_resource_pers is not None:
                    logging.info('The last resource is detected. Id={}'.format(last_resource_pers.id))
                    logging.info('Last pers_path={}'.format(last_resource_pers.last_pers_path))

                    if last_resource_pers.last_pers_path == pers_path:
                        need_create_pers_data = False
                    else:
                        logging.info('The last pers resource contains outdated data.')
                else:
                    logging.info('Any relevant pers resource is not found.')

            if need_create_recom_data:
                reasons_to_buy_path = self._get_real_path(yt_client, self.Parameters.reasons_to_buy_path)
                logging.info('Actual reasons_to_buy_path={}'.format(reasons_to_buy_path))
                last_resource_recom = MarketCsRecomData.find(attrs={"released": "stable"}).first()
                if last_resource_recom is not None:
                    logging.info('The last resource recom is detected. Id={}'.format(last_resource_recom.id))
                    logging.info('Last recom_rb_path={}'.format(last_resource_recom.last_recom_rb_path))

                    if last_resource_recom.last_recom_rb_path == reasons_to_buy_path:
                        need_create_recom_data = False
                    else:
                        logging.info('The last recom resource contains outdated data.')
                else:
                    logging.info('Any relevant recom resource is not found.')

        if not need_create_mbo_data and not need_create_pers_data and not need_create_recom_data:
            info = 'Success. The last resources already has a newest data.'
            logging.info(info)
            self.Parameters.task_status = info
            self.Parameters.has_data_prepared = False
            logging.info('All data is newest. Stop task.')
            return

        logging.info('Start creating cs data.')
        y2c = yt2csv.Yt2Csv(
            self.Parameters,
            yt_client,
            logging,
            need_create_mbo_data=need_create_mbo_data,
            need_create_pers_data=need_create_pers_data,
            need_create_recom_data=need_create_recom_data
        )
        y2c.execute()

        ts_filename = 'ts.txt'
        dt = datetime.now()
        logging.info('Start creating local arc.')

        arc_path_mbo = 'mbo_cs_data.tar.gz'
        arc_path_pers = 'pers_cs_data.tar.gz'
        arc_path_recom = 'recom_cs_data.tar.gz'
        if need_create_mbo_data:
            tar = tarfile.open(arc_path_mbo, "w:gz")
            for name in ['gl_params.csv', 'vendors.csv', 'nids.csv', 'hids.csv', 'gumoful.csv', ts_filename]:
                logging.info('Add {} to mbo data.'.format(name))
                tar.add(name, arcname=os.path.join('app/data/mbo', name))
            tar.close()
        if need_create_pers_data:
            tar = tarfile.open(arc_path_pers, "w:gz")
            logging.info('Add pers.csv to pers data.')
            tar.add('pers.csv', arcname=os.path.join('app/data/pers', 'pers.csv'))
            tar.add(ts_filename, arcname=os.path.join('app/data/pers', ts_filename))
            tar.close()
        if need_create_recom_data:
            tar = tarfile.open(arc_path_recom, "w:gz")
            logging.info('Add reasons_to_buy to recom data.')
            tar.add('reasons_to_buy', arcname=os.path.join('app/data/recom', 'reasons_to_buy'))
            tar.add(ts_filename, arcname=os.path.join('app/data/recom', ts_filename))
            tar.close()
        logging.info('Finish creating local arc.')

        resource_mbo = {}
        resource_pers = {}
        resource_recom = {}
        if self.Parameters.is_testing:
            if need_create_mbo_data:
                resource_mbo = MarketCsDataTest(self, 'Data from mbo for content-storage (testing)', 'cs_data_mbo_{}_{}_{}_{}.tar.gz'.format(dt.year, dt.day, dt.hour, dt.minute))
            if need_create_pers_data:
                resource_pers = MarketCsPersDataTest(self, 'Data from pers for content-storage (testing)', 'cs_data_pers_{}_{}_{}_{}.tar.gz'.format(dt.year, dt.day, dt.hour, dt.minute))
            if need_create_recom_data:
                resource_recom = MarketCsRecomData(self, 'Data from pers for content-storage (testing)', 'cs_data_recom_{}_{}_{}_{}.tar.gz'.format(dt.year, dt.day, dt.hour, dt.minute))
        else:
            if need_create_mbo_data:
                resource_mbo = MarketCsDataProd(self, 'Data from mbo for content-storage (production)', 'cs_data_mbo_{}_{}_{}_{}.tar.gz'.format(dt.year, dt.day, dt.hour, dt.minute))
            if need_create_pers_data:
                resource_pers = MarketCsPersDataProd(self, 'Data from pers for content-storage (production)', 'cs_data_pers_{}_{}_{}_{}.tar.gz'.format(dt.year, dt.day, dt.hour, dt.minute))
            if need_create_recom_data:
                resource_recom = MarketCsRecomData(self, 'Data from pers for content-storage (production)', 'cs_data_recom_{}_{}_{}_{}.tar.gz'.format(dt.year, dt.day, dt.hour, dt.minute))

        if need_create_mbo_data:
            resource_mbo.last_mbo_export = mbo_export_path
            data_mbo = sdk2.ResourceData(resource_mbo)

            data_mbo.path.write_bytes(spath.Path(arc_path_mbo).read_bytes())
            logging.info('Finish creating cs mbo data.')
            data_mbo.ready()
        if need_create_pers_data:
            resource_pers.last_pers_path = pers_path
            data_pers = sdk2.ResourceData(resource_pers)

            data_pers.path.write_bytes(spath.Path(arc_path_pers).read_bytes())
            logging.info('Finish creating cs pers data.')
            data_pers.ready()
        if need_create_recom_data:
            resource_recom.last_recom_rb_path = reasons_to_buy_path
            data_recom = sdk2.ResourceData(resource_recom)

            data_recom.path.write_bytes(spath.Path(arc_path_recom).read_bytes())
            logging.info('Finish creating cs recom data.')
            data_recom.ready()

        self.Parameters.has_data_prepared = True
        self.Parameters.task_status = 'Success. The resource is prepared.'

    def on_release(self, additional_parameters):
        release_integration.ReleaseToYaDeployTask2.on_release(self, additional_parameters)
