import os
import time


class DataWriter:
    def __init__(self, file_path, delimiter):
        self.file = open(file_path, 'w')
        self.delimiter = delimiter

    def write_row(self, data):
        is_first = True
        for el in data:
            if is_first:
                self.file.write(str(el).strip())
                is_first = False
            else:
                self.file.write(self.delimiter)
                self.file.write(str(el).strip())
        self.file.write('\n')

    def file_close(self):
        self.file.close()


class Yt2Csv:
    def __init__(
        self,
        params,
        yt_client,
        log,
        dir="",
        is_local_run=False,
        vendors=None,
        nids=None,
        hids=None,
        gl_params=None,
        templates=None,
        pers=None,
        reasons_to_buy=None,
        need_create_mbo_data=True,
        need_create_pers_data=True,
        need_create_recom_data=True
    ):
        self.__dir = dir
        self.__is_local_run = is_local_run
        self._log = log
        self.__need_create_mbo_data = need_create_mbo_data
        self.__need_create_pers_data = need_create_pers_data
        self.__need_create_recom_data = need_create_recom_data
        if not self.__is_local_run:
            self._params = params
            self._yt_client = yt_client
        else:
            self.__vendors = vendors or []
            self.__nids = nids or []
            self.__hids = hids or []
            self.__gl_params = gl_params or []
            self.__templates = templates or []
            self.__pers = pers or []
            self.__reasons_to_buy = reasons_to_buy or {}

    def __read_vendors(self, rows):
        from market.proto.content.mbo import MboVendors_pb2
        self._log.info('Start read vendors')
        writer = DataWriter(os.path.join(self.__dir, 'vendors.csv'), '\t||\t')
        writer.write_row(['id', 'name', 'website', 'logo'])

        for row in rows:
            result = [row['id'], row['name']]
            vendor_data = MboVendors_pb2.GlobalVendor()

            import yt.wrapper as yt
            data = yt.yson.get_bytes(row['data'])
            vendor_data.ParseFromString(data)
            site = vendor_data.site if vendor_data.site is not None else ''
            result.append(site)
            logo = vendor_data.logo[0].url if len(vendor_data.logo) > 0 and vendor_data.logo[0].url is not None else ''
            result.append(logo)

            writer.write_row(result)

        writer.file_close()

    def __read_vendors_yt(self):
        rows_iterator = self._yt_client.read_table(
            table=self._params.vendors_table_path
        )
        self.__read_vendors(rows_iterator.rows)

    def __read_vendors_local(self):
        self.__read_vendors(self.__vendors)

    def __read_nids(self, rows):
        self._log.info('Start read nids')
        writer = DataWriter(os.path.join(self.__dir, 'nids.csv'), delimiter='\t||\t')
        writer.write_row(['nid', 'hid', 'name', 'full_name', 'is_main', 'parent_id'])

        for row in rows:
            result = [row['nid'], row['hid'], row['name']]
            fullName = row['unique_name'] if row['unique_name'] is not None else row['name']
            is_main_nid = row['main']
            result.append(fullName)
            result.append(is_main_nid)
            result.append(row['parent_id'])

            writer.write_row(result)

        writer.file_close()

    def __read_nids_yt(self):
        rows_iterator = self._yt_client.read_table(
            table=self._params.nids_table_path
        )
        self.__read_nids(rows_iterator.rows)

    def __read_nids_local(self):
        self.__read_nids(self.__nids)

    def __read_hids(self, rows):
        self._log.info('Start read hids')
        writer = DataWriter(os.path.join(self.__dir, 'hids.csv'), delimiter='\t||\t')
        writer.write_row(['hid', 'name', 'full_name', 'output_type', 'is_leaf'])

        for row in rows:
            result = [
                row['hid'],
                row['name'],
                row['unique_name'],
                row['output_type'],
                row['leaf']
            ]

            writer.write_row(result)

        writer.file_close()

    def __read_hids_yt(self):
        rows_iterator = self._yt_client.read_table(
            table=self._params.hids_table_path
        )
        self.__read_hids(rows_iterator.rows)

    def __read_hids_local(self):
        self.__read_hids(self.__hids)

    def __read_gl_params(self, rows):
        self._log.info('Start read gl params')
        from market.proto.content.mbo import MboParameters_pb2
        writer = DataWriter(os.path.join(self.__dir, 'gl_params.csv'), delimiter='\t||\t')
        writer.write_row(['hid', 'id', 'name', 'xsl_name', 'is_gurulight', 'kind', 'common_filter_index', 'important', 'value_type', 'data'])

        for row in rows:
            result = [row['hid'], row['id']]

            name = row['name'] if row['name'] is not None else ''
            result.append(name)
            result.append(row['xsl_name'])

            param_data = MboParameters_pb2.Parameter()

            import yt.wrapper as yt
            data = yt.yson.get_bytes(row['data'])
            param_data.ParseFromString(data)

            result.append(param_data.use_for_gurulight)
            kind = 1 if row['param_type'] == "MODEL_LEVEL" else 2
            result.append(kind)
            result.append(row['common_filter_index'])
            result.append(param_data.important)
            # Add new values here!

            result.append(row['value_type'])

            if row['value_type'] == 'ENUM':
                for option in param_data.option:
                    if len(option.name) > 0:
                        result.append(option.id)
                        result.append(option.name[0].name)
            elif row['value_type'] == 'NUMERIC':
                unit = param_data.unit.name[0].name if len(param_data.unit.name) > 0 else ''
                result.append(unit)
                result.append(param_data.precision)
                result.append(param_data.max_value)
                min_value = param_data.min_value if param_data.min_value is not None else 0
                result.append(min_value)
            elif row['value_type'] == 'BOOLEAN':
                for option in param_data.option:
                    if len(option.name) > 0:
                        result.append(option.id)
                        result.append(option.name[0].name == "TRUE")
            elif row['value_type'] == 'NUMERIC_ENUM':
                unit = param_data.unit.name[0].name if len(param_data.unit.name) > 0 else ''
                result.append(unit)
                result.append(param_data.precision)
                result.append(param_data.max_value)
                min_value = param_data.min_value if param_data.min_value is not None else 0
                result.append(min_value)

                for option in param_data.option:
                    if len(option.name) > 0:
                        result.append(option.id)
                        result.append(option.name[0].name)

            writer.write_row(result)

        writer.file_close()

    def __read_gl_params_yt(self):
        rows_iterator = self._yt_client.read_table(
            table=self._params.params_table_path
        )
        self.__read_gl_params(rows_iterator.rows)

    def __read_gl_params_local(self):
        self.__read_gl_params(self.__gl_params)

    def __read_templates(self, rows):
        from market.proto.content.mbo import MboParameters_pb2
        self._log.info('Start read templates')

        writer = DataWriter(os.path.join(self.__dir, 'gumoful.csv'), delimiter='\t||\t')
        writer.write_row(['hid', 'model_template', 'friendly_model_template', 'seo_template', 'micro_model_template'])

        for row in rows:
            result = [row['hid']]

            category_data = MboParameters_pb2.Category()

            import yt.wrapper as yt
            data = yt.yson.get_bytes(row['data'])
            category_data.ParseFromString(data)

            model_template = category_data.model_template if category_data.model_template is not None else ''
            friendly_model_template = category_data.friendly_model_template if category_data.friendly_model_template is not None else ''
            seo_template = category_data.seo_template if category_data.seo_template is not None else ''
            micro_model_template = category_data.micro_model_template if category_data.micro_model_template is not None else ''

            result.append(model_template)
            result.append(friendly_model_template)
            result.append(seo_template)
            result.append(micro_model_template)

            writer.write_row(result)

        writer.file_close()

    def __read_templates_yt(self):
        rows_iterator = self._yt_client.read_table(
            table=self._params.gumoful_templates_table_path
        )
        self.__read_templates(rows_iterator.rows)

    def __read_templates_local(self):
        self.__read_templates(self.__templates)

    def __read_pers(self, rows):
        self._log.info('Start read perses')

        writer = DataWriter(os.path.join(self.__dir, 'pers.csv'), delimiter='\t||\t')
        writer.write_row(['model_id', 'pub_grade_count', 'rating_count', 'rating_round_01', 'rating_round_5'])

        for row in rows:
            result = [
                row['model_id'],
                row['pub_grade_count'],
                row['rating_count'],
                row['rating_round_01'],
                row['rating_round_5']
            ]

            writer.write_row(result)

        writer.file_close()

    def __read_pers_yt(self):
        rows_iterator = self._yt_client.read_table(
            table=self._params.pers_table_path
        )
        self.__read_pers(rows_iterator.rows)

    def __read_pers_local(self):
        self.__read_pers(self.__pers)

    def __read_reasons_to_buy_yt(self):
        self._log.info('Start read reasons_to_buy')
        rb_file = open(os.path.join(self.__dir, 'reasons_to_buy'), 'wb')
        rb_file.write(self._yt_client.read_file(self._params.reasons_to_buy_path).read())
        rb_file.close()

    def __read_reasons_to_buy_local(self):
        self._log.info('Start read reasons_to_buy')
        writer = DataWriter(os.path.join(self.__dir, 'reasons_to_buy'), delimiter='\t')
        writer.write_row(['model_id', 'reason_json'])

        for model_id, reasons in self.__reasons_to_buy.items():
            writer.write_row([model_id, reasons])

        writer.file_close()

    def __make_ts_file(self):
        filename = os.path.join(self.__dir, "ts.txt")
        with open(filename, 'w') as f:
            f.write(time.strftime("%Y-%m-%dT%H:%M:%S", time.gmtime()))

    # This function is also run from a test env of the content_storage_service
    def execute(self):
        self.__make_ts_file()
        if self.__is_local_run:
            self.__read_vendors_local()
            self.__read_nids_local()
            self.__read_hids_local()
            self.__read_gl_params_local()
            self.__read_templates_local()
            self.__read_pers_local()
            self.__read_reasons_to_buy_local()
        else:
            if self.__need_create_mbo_data:
                self.__read_vendors_yt()
                self.__read_nids_yt()
                self.__read_hids_yt()
                self.__read_gl_params_yt()
                self.__read_templates_yt()
            if self.__need_create_pers_data:
                self.__read_pers_yt()
            if self.__need_create_recom_data:
                self.__read_reasons_to_buy_yt()
