# -*- coding: utf-8 -*-

import logging
import tarfile
import os
from datetime import datetime
import sandbox.sdk2.path as spath
from sandbox import sdk2
from sandbox.projects.common.ya_deploy import release_integration


class MarketCsFastSettingsTest(sdk2.Resource):
    """ Market content-storage fast settings file (testing) """
    releasable = True
    auto_backup = True
    ttl = 14


class MarketCsFastSettingsProd(sdk2.Resource):
    """ Market content-storage fast settings file (production) """
    releasable = True
    auto_backup = True
    ttl = 14


class MarketCsExperimentsTest(sdk2.Resource):
    """ Market content-storage file with default experiments (testing) """
    releasable = True
    auto_backup = True
    ttl = 14


class MarketCsExperimentsProd(sdk2.Resource):
    """ Market content-storage file with default experiments (production) """
    releasable = True
    auto_backup = True
    ttl = 14


class MarketCsFastSettingsUploader(release_integration.ReleaseToYaDeployTask2, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        description = "Upload fast data for content-storage"
        is_testing = sdk2.parameters.Bool("Load testing data?", required=True)
        need_update_settings = sdk2.parameters.Bool("Upload fast-settings?", required=True)
        need_update_exp = sdk2.parameters.Bool("Upload default experiments?", required=True)
        file = sdk2.parameters.Resource("Data file", required=True)
        yp_token = sdk2.parameters.String("YP Token Secret", default='sec-01fy3w8q2q0y41hztt8vqf56p8', required=True)

    def get_yp_oauth_token(self):
        yp_secret = sdk2.yav.Secret(self.Parameters.yp_token)
        return yp_secret.data()['yp_token']

    def on_execute(self):
        logging.info('Start create resource')

        if self.Parameters.need_update_settings:
            raw_data = sdk2.ResourceData(self.Parameters.file)

            tar_name = 'fast_settings.tar.gz'
            tar = tarfile.open(tar_name, "w:gz")
            logging.info('Add settings file to archive')
            tar.add(str(raw_data.path), arcname=os.path.join('/app/data/fast_data', 'fast_settings'))
            tar.close()

            dt = datetime.now()
            if self.Parameters.is_testing:
                logging.info('Create testing fast-settings resource')
                result = sdk2.ResourceData(MarketCsFastSettingsTest(self, 'Cs fast settings (testing)', 'cs_fast_settings_{}_{}_{}_{}.tar.gz'.format(dt.year, dt.day, dt.hour, dt.minute)))
                result.path.write_bytes(spath.Path(tar_name).read_bytes())
                result.ready()
            else:
                logging.info('Create production fast-settings resource')
                result = sdk2.ResourceData(MarketCsFastSettingsProd(self, 'Cs fast settings (production)', 'cs_fast_settings_{}_{}_{}_{}.tar.gz'.format(dt.year, dt.day, dt.hour, dt.minute)))
                result.path.write_bytes(spath.Path(tar_name).read_bytes())
                result.ready()
        elif self.Parameters.need_update_exp:
            data = sdk2.ResourceData(self.Parameters.file)
            dt = datetime.now()
            if self.Parameters.is_testing:
                logging.info('Create testing default experiments resource')
                result = sdk2.ResourceData(MarketCsExperimentsTest(self, 'Cs default experiments (testing)', 'cs_default_experiments_{}_{}_{}_{}'.format(dt.year, dt.day, dt.hour, dt.minute)))
                result.path.write_bytes(spath.Path(data.path).read_bytes())
                result.ready()
            else:
                logging.info('Create production default experiments resource')
                result = sdk2.ResourceData(MarketCsExperimentsProd(self, 'Cs default experiments (prod)', 'cs_default_experiments_{}_{}_{}_{}'.format(dt.year, dt.day, dt.hour, dt.minute)))
                result.path.write_bytes(spath.Path(data.path).read_bytes())
                result.ready()
        else:
            logging.info('The data type to update is not selected. Stop task')

    def on_release(self, additional_parameters):
        release_integration.ReleaseToYaDeployTask2.on_release(self, additional_parameters)
