# -*- coding: utf-8 -*-

import logging
import requests
import time
from datetime import datetime
import sandbox.sdk2.path as spath
from sandbox import sdk2
from sandbox.projects.common.ya_deploy import release_integration


class MarketCsNavTreeTesting(sdk2.Resource):
    """ Market content-storage navigation trees file (testing) """
    releasable = True
    auto_backup = True
    ttl = 14


class MarketCsNavTreeProd(sdk2.Resource):
    """ Market content-storage navigation trees file (production) """
    releasable = True
    auto_backup = True
    ttl = 14


class MarketCsMboDataDownloader(release_integration.ReleaseToYaDeployTask2, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        description = "Download data from mbo and create sb resource"
        is_testing = sdk2.parameters.Bool("Load testing data?", required=True)
        need_navigation = sdk2.parameters.Bool("Load navigation data?", required=True)
        yp_token = sdk2.parameters.String("YP Token Secret", default='sec-01fy3w8q2q0y41hztt8vqf56p8', required=True)

        with sdk2.parameters.Output():
            has_data_prepared = sdk2.parameters.Bool("Task has a data prepared", default=False)
            task_status = sdk2.parameters.String("Task status", default='Unknown error')

    def get_yp_oauth_token(self):
        yp_secret = sdk2.yav.Secret(self.Parameters.yp_token)
        return yp_secret.data()['yp_token']

    def on_execute(self):
        logging.info('Start MarketCsMboDataDownloader')

        need_any_data = any(
            [
                self.Parameters.need_navigation,
            ]
        )

        ts_str = time.strftime("%Y-%m-%dT%H:%M:%S", time.gmtime())
        if self.Parameters.need_navigation:
            host = 'mbo-http-exporter.tst.vs.market.yandex.net:8084' if self.Parameters.is_testing else 'mbo-http-exporter.yandex.net:8084'
            download_url = 'http://{}/navigation?skip-validation=true'.format(host)
            retry_count = 3

            response = {}
            ok = False
            logging.info('Start download navigation tree from {}'.format(download_url))
            for i in range(retry_count):
                logging.info('Download attempt {}'.format(i + 1))
                try:
                    # Сейчас этот запрос отрабатывает +- 20 минут
                    response = requests.get(download_url, timeout=1800)
                except Exception as ex:
                    logging.warn('Cant do request to {} because of exception: {}'.format(host, ex.message))
                    continue
                if response.status_code != 200:
                    logging.warn('Cant do request to {}, status code == {}'.format(host, response.status_code))
                    continue
                ok = True
                break

            if not ok:
                raise NotImplementedError('Cant do request to {} for {} times'.format(host, retry_count))

            file_name = 'navigation.xml'
            nav_f = open(file_name, 'w')
            nav_f.write(response.text)
            nav_f.write(ts_str)
            nav_f.close()

            dt = datetime.now()
            if self.Parameters.is_testing:
                logging.info('Create testing navigation tree resource')
                result = sdk2.ResourceData(MarketCsNavTreeTesting(
                    self,
                    'Cs navigation tree (testing)',
                    'cs_navigation_{}_{}_{}_{}'.format(dt.year, dt.day, dt.hour, dt.minute)
                ))
                result.path.write_bytes(spath.Path(file_name).read_bytes())
                result.ready()
            else:
                logging.info('Create production navigation tree resource')
                result = sdk2.ResourceData(MarketCsNavTreeProd(
                    self,
                    'Cs navigation tree (production)',
                    'cs_navigation_{}_{}_{}_{}'.format(dt.year, dt.day, dt.hour, dt.minute)
                ))
                result.path.write_bytes(spath.Path(file_name).read_bytes())
                result.ready()

        if not need_any_data:
            self.Parameters.task_status = 'Success. MarketCsMboDataDownloader should not uploaded any data'
            self.Parameters.has_data_prepared = False
        else:
            self.Parameters.task_status = 'Success. MarketCsMboDataDownloader uploaded all data'
            self.Parameters.has_data_prepared = True

    def on_release(self, additional_parameters):
        release_integration.ReleaseToYaDeployTask2.on_release(self, additional_parameters)
