import os
import io
import logging
import re

import jinja2
from sandbox import sdk2
from sandbox.projects.common import vcs
from sandbox.projects.market.infra.MarketApplicationArcProducerBase import MarketApplicationArcProducerBase


class CreateBaseExp3Config(MarketApplicationArcProducerBase):
    arc_token = None

    class Parameters(sdk2.Task.Parameters):
        ya_owner = sdk2.parameters.String('Ya owner', required=True)
        config_name = sdk2.parameters.String('Config name', required=True)
        path = sdk2.parameters.String("Dir from acradia root to create config files", required=True)
        is_create_dirs = sdk2.parameters.Bool("If any dir in path is not existed - we will create it",
                                              required=False, default=True)
        is_py_proto = sdk2.parameters.Bool("If you have python app turn on py proto", required=False, default=False)

        commit_message = sdk2.parameters.String("Commit message", required=False, default="exp3 config produced")
        arc_secret_owner = sdk2.parameters.String("Secret owner (for SB Vault)", default="MARKET", required=False)
        arc_secret_id = sdk2.parameters.String("Vault secret ID", default="", required=False)
        arc_secret_key = sdk2.parameters.String("Secret key", default="ARC_TOKEN", required=True)
        arcadia_user = sdk2.parameters.String('Arcadia user', required=True, default='robot-market-infra')

    @property
    def base_branch_name(self):
        return 'MarketExp3ConfigProducer'

    @property
    def _full_branch_name(self):
        full_revision = "\n".join(self.arc.describe(self.arcadia_root, svn=True))
        revision = full_revision.split("-")[0].replace("r", "")
        return "{}-{}-{}".format(self.base_branch_name, self.id, revision)

    def on_execute(self):
        self.try_get_arc_token()

        if self.arc_token:
            self._run_task()
        else:
            raise RuntimeError('Arc token doesn\'t exist')

    def _run_task(self):
        self.arc = vcs.arc.Arc(arc_oauth_token=self.arc_token)

        with self.arc.mount_path("", changeset="trunk", fetch_all=False) as arcadia_root:
            self.arcadia_root = arcadia_root
            should_commit = self.produce_base_config()
            if should_commit:
                self.push_changes()

    def produce_base_config(self):
        logging.info("Starting produce base config..")

        config_name = str(self.Parameters.config_name)
        full_path = os.path.join(self.arcadia_root, str(self.Parameters.path))
        proto_path = os.path.join(full_path, 'schema.proto')
        values_path = os.path.join(full_path, 'values.yaml')
        values_local_path = os.path.join(full_path, 'values-overrides.local.yaml')
        values_production_path = os.path.join(full_path, 'values-overrides.production.yaml')
        values_testing_path = os.path.join(full_path, 'values-overrides.testing.yaml')
        config_ya_make_path = os.path.join(full_path, 'ya.make')

        if bool(self.Parameters.is_create_dirs):
            logging.info("Collect folders to modify ya make..")
            ya_makes = {}
            current_path = full_path
            while not os.path.exists(current_path):
                prev_path = os.path.dirname(current_path)
                current_fold = os.path.basename(current_path)
                if prev_path == '' or prev_path == '/':
                    raise RuntimeError('Base dir must exist')
                ya_makes[prev_path] = current_fold
                current_path = prev_path

            logging.info("Creating not existed dirs..")
            if not os.path.exists(full_path):
                os.makedirs(full_path)
            elif self.is_proto_exist(proto_path, config_name):
                logging.info("schema proto with given config_name already exist. finishing task with success")
                return False
            elif os.listdir(full_path):
                raise RuntimeError('Dir is not empty')
            logging.info("Creating ya.makes")
            for key in ya_makes:
                logging.info(key + ": " + "RECURSE(" + ya_makes[key] + ")")
                with open(os.path.join(key, 'ya.make'), 'a') as f:
                    f.write("\nRECURSE(" + ya_makes[key] + ")")
        elif not os.path.exists(full_path):
            raise RuntimeError('Can not creat config files via {} not existed and is_create_dirs=False', full_path)
        elif self.is_proto_exist(proto_path, config_name):
            logging.info("schema proto with given config_name already exist. finishing task with success")
            return False
        elif os.listdir(full_path):
            raise RuntimeError('Dir is not empty')

        jinja_schema_path = os.path.join(self.arcadia_root,
                                         'sandbox/projects/market/devexp/exp3/create_base_config/schema.proto.jinja2')
        jinja_config_ya_make_path = os.path.join(self.arcadia_root,
                                                 'sandbox/projects/market/devexp/exp3',
                                                 'create_base_config/config_ya_make.jinja2')

        logging.info("read jinja2 schema template")
        with io.open(jinja_schema_path, "r", encoding="utf-8") as file:
            tmpl = file.read()
        logging.info("create schema.proto:")
        with open(proto_path, 'w') as proto_file:
            template_context = {
                "full_path": str(self.Parameters.path),
                "config_name": config_name,
            }
            tmpl_processed = jinja2.Template(tmpl, keep_trailing_newline=True).render(template_context)
            logging.info(tmpl_processed)
            proto_file.write(tmpl_processed)
        logging.info("create values.yaml")
        with open(values_path, 'w') as values_file:
            values_file.write("")
        with open(values_local_path, 'w') as values_file:
            values_file.write("")
        with open(values_production_path, 'w') as values_file:
            values_file.write("")
        with open(values_testing_path, 'w') as values_file:
            values_file.write("")
        logging.info("create config ya.make:")
        with io.open(jinja_config_ya_make_path, "r", encoding="utf-8") as file:
            tmpl = file.read()
        with open(config_ya_make_path, 'w') as config_ya_make:
            template_context = {
                "ya_owner": str(self.Parameters.ya_owner),
                "is_py_proto": bool(self.Parameters.is_py_proto)
            }
            tmpl_processed = jinja2.Template(tmpl, keep_trailing_newline=True).render(template_context)
            logging.info(tmpl_processed)
            config_ya_make.write(tmpl_processed)
        return True

    def is_proto_exist(self, proto_path, config_name):
        if os.path.exists(proto_path):
            with io.open(proto_path, 'r', encoding="utf-8") as proto_file:
                groups = re.search(r"(.)*\(exp3_config_name\)(.)*\"(.*)\"", proto_file.read())
                if groups is None or groups.group(3) != config_name:
                    raise RuntimeError('schema.proto with another config name already exists')
            return True
        return False
