# coding=utf-8

import os
import logging

from sandbox import sdk2
from sandbox.common.types import misc as ctm
from sandbox.projects.market.devexp.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.devexp.front.helpers.node import create_node_selector
from sandbox.projects.market.devexp.front.helpers.sandbox_helpers import get_resource_http_proxy_link as get_resource_link
from sandbox.projects.market.devexp.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.projects.market.devexp.front.helpers.SetEnv import SetEnv
from sandbox.projects.market.devexp.front.helpers.startrack import ReleaseTicket
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.common.arcadia import sdk as arcadia_sdk


ST_OAUTH_TOKEN_VAULT_KEY = "robot-market-devexp-st-token"
DISK_SPACE = 3 * 1024  # 3 Gb


class MARKET_DEVEXP_AUTOTEST_REPORT(sdk2.Resource):
    """Allure autotest report"""

    any_arch = True
    __default_attribute__ = sdk2.parameters.String


class MarketDevexpAutotestsHermione(sdk2.Task):
    """
    Hermione автотесты планнера
    """

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = DISK_SPACE
        environments = [
            PipEnvironment("yandex_tracker_client", version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment("startrek_client", version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    class Parameters(sdk2.Task.Parameters):
        ubuntu_version = create_ubuntu_selector()
        nodejs_version = create_node_selector()

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia repository url",
            default_value="arcadia-arc:/#trunk",
            required=True
        )
        app_dir_with_arc = sdk2.parameters.String(
            "App dir",
            default="market/infra/market-planner/webapp",
            required=True
        )
        run_bash_script = sdk2.parameters.String(
            "Путь до запускаемого bash скрипта",
            default="./scripts/hermione.bash",
            required=True
        )
        base_url = sdk2.parameters.String(
            "Тестируемый хост",
            description="Адрес машины, на который будут ходить тесты",
            required=True
        )
        grid_url = sdk2.parameters.String(
            "Селениум грид",
            default_value="http://market:a1aa51bb39bb728ca365e421d13dd14c@sg.yandex-team.ru:4444/wd/hub",
            required=True
        )
        issue = ReleaseTicket(
            "Релизный тикет",
            description='Пример: "MARKETQPLANNER-1444"'
        )

    def on_enqueue(self):
        super(MarketDevexpAutotestsHermione, self).on_enqueue()
        setup_container(self)

    def _prepare_dirs(self):
        self.report_dir = self.path("report")

    def _prepare_env(self):
        os.environ["GRID_URL"] = self.Parameters.grid_url
        os.environ["BASE_URL"] = self.Parameters.base_url
        os.environ["REPORT_DIR"] = str(self.report_dir)
        os.environ["CI_SYSTEM"] = "True"

    def _process_task(self, app_path):
        rich_check_call(["bash", "-c", self.Parameters.run_bash_script], self, "test", cwd=app_path)

    def _create_report(self):
        resource = MARKET_DEVEXP_AUTOTEST_REPORT(self, "Report", self.report_dir)
        sdk2.ResourceData(resource).ready()

        return resource

    def _send_startrek_report(self, resource):
        if not self.Parameters.issue:
            logging.debug("Release ticket not specified")
            return

        report_url = "{}/index.html".format(get_resource_link(resource)) if resource else ""
        oauth_token = sdk2.Vault.data(ST_OAUTH_TOKEN_VAULT_KEY)

        from startrek_client import Startrek
        st = Startrek(useragent="robot-market-devexp", token=oauth_token)
        issue = st.issues[self.Parameters.issue]
        # TODO: Добавить в коммент статистику прогона
        issue.comments.create(text="Отчет выполнения e2e тестов: {}".format(report_url))

    def on_prepare(self):
        self._prepare_dirs()
        self._prepare_env()

    def on_execute(self):
        super(MarketDevexpAutotestsHermione, self).on_execute()
        with SetEnv(self, nodejs_version=self.Parameters.nodejs_version, arc_token_vault="robot-market-devexp-arc-token"):
            self._prepare_repos()

    def _prepare_repos(self):
        url = self.Parameters.arcadia_url
        use_arc = True
        with arcadia_sdk.mount_arc_path(url, use_arc_instead_of_aapi=use_arc) as arcadia_root:
            arcadia_path = os.path.join(arcadia_root, self.Parameters.app_dir_with_arc)

            try:
                self._process_task(arcadia_path)
            finally:
                report_resource = self._create_report()
                self._send_startrek_report(report_resource)
