# coding=utf-8
import os
import tempfile
import distutils

from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox import sdk2
import sandbox.common.types.misc as ctm
import sandbox.sdk2.path as spath
from sandbox.projects.common.nanny import nanny

from sandbox.projects.market.devexp.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.devexp.front.helpers.SetEnv import SetEnv
from sandbox.projects.market.devexp.front.helpers.node import create_node_selector
from sandbox.projects.market.devexp.front.helpers.ubuntu import create_ubuntu_selector, setup_container


class MarketDevexpFrontBuildAbstract(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Abstract task to build static front
    """

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Приложение") as app:
            arcadia_url = sdk2.parameters.ArcadiaUrl(
                "Arcadia repository url",
                default_value="arcadia-arc:/#trunk",
                required=True
            )
            app_dir_with_arc = sdk2.parameters.String(
                "App dir",
                default="market/infra/market-planner/webapp",
                required=True
            )
            dist_path = sdk2.parameters.String(
                "Путь от корня проекта, куда собирается проект",
                default="webapp",
                required=True
            )
            run_bash_script = sdk2.parameters.String(
                "Путь до запускаемого bash скрипта",
                default="./scripts/build.bash",
                required=True
            )
            create_jar_resource = sdk2.parameters.Bool("Собирать в jar resource")
            with create_jar_resource.value[True]:
                jar_resource_name = sdk2.parameters.String(
                    "Название jar ресурса",
                    default="webapp.jar",
                    required=True
                )

            project_name = sdk2.parameters.String("Название проекта, попадет в env при сборке")

        ubuntu_version = create_ubuntu_selector()
        nodejs_version = create_node_selector()

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        ram = 4096

    @property
    def resource_type(self):
        """
        Тип ресурса для архива с приложением
        """
        raise NotImplementedError

    def on_enqueue(self):
        super(MarketDevexpFrontBuildAbstract, self).on_enqueue()
        setup_container(self)

    def on_execute(self):
        super(MarketDevexpFrontBuildAbstract, self).on_execute()
        with SetEnv(self, nodejs_version=self.Parameters.nodejs_version, arc_token_vault="robot-market-devexp-arc-token"):
            self._prepare_repos()

    def _build_project(self, app_path):
        project_name = self.Parameters.project_name

        if project_name:
            os.environ["PROJECT_NAME"] = project_name

        rich_check_call(
            [self.Parameters.run_bash_script],
            self,
            "build",
            env=os.environ,
            cwd=app_path
        )

    def _process_task(self, app_path):
        self._build_project(app_path)
        pack_dist_archive_path = self._create_tar(app_path)
        self._app_create_resource(pack_dist_archive_path)

    def _create_tar(self, app_path):
        dist_path = os.path.join(app_path, self.Parameters.dist_path)
        pack_dist_path = tempfile.mkdtemp()

        if self.Parameters.create_jar_resource:
            rich_check_call(
                ["jar", "-cf", pack_dist_path + "/" + self.Parameters.jar_resource_name, "-C", app_path, self.Parameters.dist_path],
                self,
                "java"
            )
        else:
            distutils.dir_util.copy_tree(str(dist_path), str(pack_dist_path))

        pack_dist_archive_path = tempfile.mktemp(suffix=".tar.gz", prefix="webapp")
        rich_check_call(
            ["tar", "-C", pack_dist_path, "-czf", pack_dist_archive_path, "."],
            self,
            "create_dist_pack"
        )

        return pack_dist_archive_path

    def _app_create_resource(self, pack_dist_archive_path):
        res = sdk2.ResourceData(self.resource_type(
            self, "Resource", "webapp.tar.gz"
        ))
        if res.path.exists():
            res.path.unlink()
        res.path.write_bytes(spath.Path(pack_dist_archive_path).read_bytes())
        res.ready()

    def _prepare_repos(self):
        url = self.Parameters.arcadia_url
        use_arc = True
        with arcadia_sdk.mount_arc_path(url, use_arc_instead_of_aapi=use_arc) as arcadia_root:
            app_path = os.path.join(arcadia_root, self.Parameters.app_dir_with_arc)
            self._process_task(app_path)
