# coding=utf-8
import datetime
import logging
import time

import sandbox.common.types.misc as ctm
from sandbox.projects.market.devexp.front.helpers.startrack import send_startrek_report
from sandbox.projects.market.devexp.front.helpers.tsum import TSUM_TOKEN_VAULT_KEY, TSUM_MT_PAGE_URL, TSUM_GET_MT_API_URL
from sandbox import sdk2
from sandbox.sdk2 import Task, Vault
from sandbox.projects.sandbox_ci.utils.request import send_request
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.sandboxsdk.environments import PipEnvironment


class MarketDevexpFrontCheckMT(Task):
    """
    Таска для проверки мультитестинга
    """

    class Requirements(Task.Requirements):
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment("yandex_tracker_client", version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment("startrek_client", version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    class Parameters(Task.Parameters):
        ubuntu_version = create_ubuntu_selector()

        ticket_id = sdk2.parameters.String(
            "Тикет",
            required=True
        )

        review_id = sdk2.parameters.String(
            "Номер пулл реквеста",
            required=True
        )

        tsum_project = sdk2.parameters.String(
            "Название проекта",
            default_value="market-planner",
            required=True
        )

        polling_timeout_minutes = sdk2.parameters.Float(
            "Таймаут поллинга в минутах",
            default_value=30.0,
            required=True
        )

        polling_interval_seconds = sdk2.parameters.Integer(
            "Интервал поллинга в секундах",
            default_value=30,
            required=True
        )

    def on_enqueue(self):
        super(MarketDevexpFrontCheckMT, self).on_enqueue()
        setup_container(self)

    @property
    def _tsum_mt_name(self):
        return self.Parameters.ticket_id.lower() + "-" + self.Parameters.review_id

    @property
    def _tsum_api_url(self):
        return TSUM_GET_MT_API_URL.format(
            project=self.Parameters.tsum_project,
            name=self._tsum_mt_name
        )

    @property
    def _tsum_page_url(self):
        return TSUM_MT_PAGE_URL.format(
            project=self.Parameters.tsum_project,
            name=self._tsum_mt_name
        )

    def _poll_mt_status(self):
        token = Vault.data(TSUM_TOKEN_VAULT_KEY)

        headers = {
            "Authorization": token,
            "Content-Type": "application/json"
        }

        end_time = datetime.datetime.now() + datetime.timedelta(minutes=self.Parameters.polling_timeout_minutes)

        while datetime.datetime.now() < end_time:
            res = send_request("get", self._tsum_api_url, headers=headers)
            res.raise_for_status()

            try:
                res_json = res.json()
                assert res_json['name'] == self._tsum_mt_name
            except Exception:
                logging.error(u"Bad TSUM response: {}".format(res.text))
                send_startrek_report("Ошибка при запуске среды: {}".format(res.text), self.Parameters.ticket_id)

                raise Exception("Something is wrong, see logs")

            status = res_json['status']

            if status == "DEPLOY_FAILED":
                send_startrek_report(
                    "Мультитестинговая среда упала: {}".format(self._tsum_page_url), self.Parameters.ticket_id
                )
                raise Exception("Мультитестинговая среда упала: {}".format(self._tsum_page_url))

            if status == "READY":
                send_startrek_report(
                    "Мультитестинговая среда успешно развернута: {}".format(self._tsum_page_url),
                    self.Parameters.ticket_id
                )
                return

            time.sleep(self.Parameters.polling_interval_seconds)

    def on_execute(self):
        self._poll_mt_status()
