# coding=utf-8

import logging
import json

import sandbox.common.types.misc as ctm
from sandbox.projects.market.devexp.front.helpers.startrack import send_startrek_report
from sandbox.projects.market.devexp.front.helpers.tsum import TSUM_CREATE_OR_RESTART_MT_API_URL, TSUM_TOKEN_VAULT_KEY,\
    TSUM_MT_PAGE_URL
from sandbox import sdk2
from sandbox.sdk2 import Task, Vault
from sandbox.projects.sandbox_ci.utils.request import send_request
from sandbox.projects.market.front.helpers.ubuntu import create_ubuntu_selector, setup_container
from sandbox.sandboxsdk.environments import PipEnvironment


class MarketDevexpFrontStartMT(Task):
    """
    Таска для запуска и перезапуска мультитестинга планера.
    """

    class Requirements(Task.Requirements):
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment("yandex_tracker_client", version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            PipEnvironment("startrek_client", version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"])
        ]

    class Parameters(Task.Parameters):
        ubuntu_version = create_ubuntu_selector()

        ticket_id = sdk2.parameters.String(
            "Тикет",
            required=True
        )

        review_id = sdk2.parameters.String(
            "Номер пулл реквеста",
            required=True
        )

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia repository url",
            default_value="arcadia-arc:/#trunk",
            required=True
        )

        tsum_project = sdk2.parameters.String(
            "Название проекта",
            default_value="market-planner",
            required=True
        )

        tsum_pipeline_type = sdk2.parameters.String(
            "Тип пайплайна",
            default_value="mt-market-planner",
            required=True
        )

    def on_enqueue(self):
        super(MarketDevexpFrontStartMT, self).on_enqueue()
        setup_container(self)

    @property
    def _tsum_mt_name(self):
        return self.Parameters.ticket_id.lower() + "-" + self.Parameters.review_id

    @property
    def _tsum_api_url(self):
        return TSUM_CREATE_OR_RESTART_MT_API_URL.format(
            project=self.Parameters.tsum_project,
            name=self._tsum_mt_name
        )

    @property
    def _tsum_page_url(self):
        return TSUM_MT_PAGE_URL.format(
            project=self.Parameters.tsum_project,
            name=self._tsum_mt_name
        )

    def _create_or_restart_mt(self):
        token = Vault.data(TSUM_TOKEN_VAULT_KEY)

        headers = {
            "Authorization": token,
            "Content-Type": "application/json"
        }

        data = {
            "name": self._tsum_mt_name,
            "type": "USE_EXISTING_PIPELINE",
            "pipelineId": self.Parameters.tsum_pipeline_type,
            "resources": {
                "32fa4237-4246-46a8-84b8-9df3efdd8301": {
                    "key": self.Parameters.ticket_id
                },
                "c549477a-867a-483f-947b-5fd64feef256": {
                    "ref": self.Parameters.arcadia_url
                },
                "fd4f152f-9f11-4b91-8a35-e58503b6bdf6": {
                    "tickets": [
                        self.Parameters.ticket_id
                    ]
                }
            }
        }

        res = send_request("post", self._tsum_api_url, headers=headers, data=json.dumps(data))

        res.raise_for_status()

        try:
            assert res.json()['name'] == self._tsum_mt_name

            # TODO: Добавить ссылки на балансеры
            send_startrek_report("Ссылка на среду: {}".format(self._tsum_page_url), self.Parameters.ticket_id)
        except Exception:
            logging.error(u"Bad TSUM response: {}".format(res.text))
            send_startrek_report("Ошибка при запуске среды: {}".format(res.text), self.Parameters.ticket_id)

            raise Exception("Something is wrong, see logs")

    def on_execute(self):
        self._create_or_restart_mt()
