# coding=utf-8
import os
import hashlib
import logging

import mimetypes

mime = mimetypes.MimeTypes()
mime.encodings_map['.br'] = 'br'


class ObjectAlreadyExistsError(Exception):
    def __init__(self, bucket_name, object_key, expected_etag, actual_etag):
        super(ObjectAlreadyExistsError, self) \
            .__init__("S3: Object at key '{}' already existst in bucket {};"
                      "Expected etag is '{}'; actual etag is '{}'".format(
                           object_key,  # noqa
                           bucket_name,
                           expected_etag,
                           actual_etag,
                      ))


def get_connection():
    from boto.s3.connection import S3Connection

    return S3Connection(
        host="s3.mds.yandex.net",
        aws_access_key_id=os.environ["AWS_ACCESS_KEY_ID"],
        aws_secret_access_key=os.environ["AWS_SECRET_ACCESS_KEY"])


def _md5(fname):
    hash_md5 = hashlib.md5()
    with open(fname, "rb") as f:
        for chunk in iter(lambda: f.read(4096), b""):
            hash_md5.update(chunk)
    return hash_md5.hexdigest()


def check_same_file(conn, bucket_name, file_path, s3_path, ignore_mismatch=False):
    """
    Возвращает True, если файл есть, и md5 совпадает
    False, если файла нет
    Бросает ObjectAlreadyExistsError, если файл есть и md5 отличачется
    """
    bucket = conn.get_bucket(bucket_name)

    s3_key = bucket.get_key(s3_path)
    if not s3_key:
        return False

    expected_etag = '"{}"'.format(_md5(file_path))

    if expected_etag != s3_key.etag:
        if ignore_mismatch:
            logging.warn("Etag mismatch! File: {}. Expected: {}. Actual: {}".format(
                file_path, expected_etag, s3_key.etag
            ))
        else:
            raise ObjectAlreadyExistsError(bucket_name, s3_path, expected_etag, s3_key.etag)

    return True


def upload_file(conn, bucket_name, file_path, s3_path):
    bucket = conn.get_bucket(bucket_name)
    key = bucket.new_key(s3_path)

    (mimeType, encType) = mime.guess_type(file_path)

    key.set_contents_from_filename(file_path, headers={
        "Content-Type": mimeType,
        "Content-Encoding": encType
    })

    logging.info("S3: Uploaded object '{}' into bucket '{}' with content-type '{}' and content-encoding '{}'".format(s3_path, bucket_name, mimeType, encType))


def safe_upload(conn, bucket_name, file_path, s3_path, ignore_mismatch=False):
    same_file_exists = check_same_file(conn, bucket_name, file_path, s3_path, ignore_mismatch)

    if not same_file_exists:
        upload_file(conn, bucket_name, file_path, s3_path)
