import hashlib
import json
import tarfile
import time
import logging
import requests

from pprint import pformat

CHUNK_SIZE_BYTES = 1048576  # 1 MByte
RETRY_COUNT = 5
RETRY_PAUSE_SECONDS = 10

TSUM_TOKEN_VAULT_KEY = "robot-metatron-tsum-token"
TSUM_MT_PAGE_URL = "https://tsum.yandex-team.ru/pipe/projects/{project}/multitestings/environments/{name}"
TSUM_GET_MT_API_URL = "https://tsum.yandex-team.ru/api/multitestings/project/{project}/environments/{name}/"
TSUM_CREATE_OR_RESTART_MT_API_URL = TSUM_GET_MT_API_URL + "createOrRestart"


def send_report_to_tsum(archive, tsum_api_hosts, sandbox_resource_type, sandbox_ticket_num, sandbox_resource_id=None):
    if not tsum_api_hosts or not len(tsum_api_hosts):
        raise RuntimeError("Parameters tsum_api_hosts must be sets")

    headers = {'Content-type': 'application/json'}
    report_json = json.dumps(_build_md5_report(archive, sandbox_resource_type, sandbox_ticket_num, sandbox_resource_id))

    for tsum_api_host in tsum_api_hosts:
        for i in range(RETRY_COUNT):
            try:
                logging.debug("Send report to TSUM host %s", tsum_api_host)

                report_url = tsum_api_host + '/agent/addResourceInfo'
                response = requests.post(url=report_url, data=report_json, headers=headers)

                response.raise_for_status()
                # TODO exit from loop on successful response
                break
            except StandardError as e:
                logging.info("Error %s", str(e.message))

                if i == RETRY_COUNT - 1:
                    raise RuntimeError("Can't send report to tsum")
                else:
                    time.sleep(i * RETRY_PAUSE_SECONDS)


def _build_md5_report(archive_path, sandbox_resource_type, sandbox_task_id, sandbox_resource_id):

    logging.debug("Build report for %s", archive_path)

    checksums = []
    tar = tarfile.open(archive_path)
    members = filter(lambda m: m.isfile(), tar.getmembers())

    for sub_file in members:
        sub_file = tar.extractfile(sub_file)

        if sub_file:
            checksums.append({
                "path": sub_file.name,
                "md5": _get_md5(sub_file)
            })
            sub_file.close()

    tar.close()

    with open(archive_path) as tar_archive:
        md5_report = {
            "resource": {
                "resourceType": sandbox_resource_type,
                "resourceId": sandbox_resource_id,
                "taskId": sandbox_task_id
            },
            "md5": _get_md5(tar_archive),
            "checksums": checksums
        }

    logging.debug("Report body %s", pformat(md5_report))

    return md5_report


def _get_md5(check_file):
    hash_md5 = hashlib.md5()

    while True:
        file_bytes = check_file.read(CHUNK_SIZE_BYTES)
        if file_bytes:
            hash_md5.update(file_bytes)
        else:
            break

    return hash_md5.hexdigest()
