# -*- coding: utf-8 -*-
import os
import logging
import datetime
import requests

from sandbox import sdk2
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
from sandbox.common.telegram import TelegramBot


class MarketDynamicPricingApp(sdk2.Resource):
    """Dynamic pricing application (stable)"""

    executable = True


class MarketDynamicPricingAppTest(sdk2.Resource):
    """Dynamic pricing application (testing)"""

    executable = True


CUSTOM_ENVIRONMENT = "custom"


class RunMarketDynamicPricing(sdk2.Task):
    """Calculate prices for 1P goods for beru.ru
    """

    class Parameters(sdk2.Task.Parameters):
        oauth_token = sdk2.parameters.Vault("Vault secret contains OAuth token; should provide access to YT and STEP", required=True)
        use_latest_version = sdk2.parameters.Bool("Use latest version of dynamic pricing application", default=True)
        with use_latest_version.value[False]:
            app_resource = sdk2.parameters.Resource(
                "Resource with dynamic pricing application",
                resource_type=MarketDynamicPricingApp,
                state=ctr.State.READY,
                required=True
            )

        with sdk2.parameters.RadioGroup("Package build type") as build_type:
            build_type.values.custom = build_type.Value(CUSTOM_ENVIRONMENT, default=True)
            build_type.values.testing = build_type.Value(ctt.ReleaseStatus.TESTING)
            build_type.values.production = build_type.Value(ctt.ReleaseStatus.STABLE)
        with sdk2.parameters.RadioGroup("Environment for task") as environment:
            environment.values.custom = environment.Value(CUSTOM_ENVIRONMENT, default=True)
            environment.values.testing = environment.Value("testing")
            environment.values.production = environment.Value("production")
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            cluster = sdk2.parameters.String("YT cluster", required=True, default="hahn")
            with environment.value[CUSTOM_ENVIRONMENT]:
                config_path = sdk2.parameters.String("Path to config in YT", required=True)
                yt_dest_directory = sdk2.parameters.String("YT directory", required=True)
                input_erp_table = sdk2.parameters.String("Erp Autostrategy stats table", required=True)
            input_demand_dir = sdk2.parameters.String("Elasticity dir (without latest)", default="")
        with sdk2.parameters.Group('Notification parameters') as notify_parameters:
            step_event = sdk2.parameters.Bool("Send STEP event", default=False)
            telegram_notification = sdk2.parameters.Bool("Send report to Telegram", default=False)
            with telegram_notification.value[True]:
                telegram_send_on_success = sdk2.parameters.Bool("Send report on SUCCESS", default=False)
                telegram_chat_id = sdk2.parameters.Integer("Telegram chat ID", required=True)
                telegram_bot_token_vault = sdk2.parameters.Vault("Vault secret contains Telegram bot token", required=True)
        with sdk2.parameters.Group('STEP parameters') as step_parameters:
            event_params = sdk2.parameters.Dict("Params from STEP event, overrides all")

    def _notify(self, status):
        if not self.Parameters.telegram_notification:
            return
        if not self.Parameters.telegram_send_on_success and status == ctt.Status.SUCCESS:
            return

        link = 'https://sandbox.yandex-team.ru/task/{}/view'.format(self.id)
        message = "Sandbox dynamic pricing task #{} has finished: {}\nLink: {}".format(
            self.id,
            status,
            link
        )
        try:
            bot = TelegramBot(bot_token=self.Parameters.telegram_bot_token_vault.data())
            bot.send_message(self.Parameters.telegram_chat_id, message)
        except Exception as e:
            logging.warn("Telegram notification failed: {}".format(e.message))

    def on_finish(self, prev_status, status):
        self._notify(status)

    def on_break(self, prev_status, status):
        self._notify(status)

    def on_execute(self):
        # get data and application
        app_path = self._get_app()

        timestamp = self.cur_ts
        cur_ts = timestamp.strftime('%Y-%m-%dT%H:%M:%S')
        today = timestamp.strftime("%Y-%m-%d")

        output_prices_path = "{}/output/prices".format(self.yt_dest_directory)
        output_margins_path = "{}/output/margins".format(self.yt_dest_directory)
        output_check_path = "{}/output/check".format(self.yt_dest_directory)
        output_metrics_path = "{}/output/metrics".format(self.yt_dest_directory)
        output_stock_check_path = "{}/output/stock_check".format(self.yt_dest_directory)

        logging.info((
            "\nApplication path: {}"
            "\nYT cluster: {}"
            "\nPath to configcluster: {}"
            "\nInput erp table: {}"
            "\nInput demand table: {}"
            "\nOutput prices path: {}"
            "\nOutput margins path: {}"
            "\nOutput check path: {}"
            "\nOutput metrics path: {}"
            "\nOutput stock check path: {}"
            "\nDate: {}"
            "\nTimestamp: {}"
        ).format(app_path, self.cluster, self.config_path,  self.input_erp_table, self.input_demand_table,
            output_prices_path, output_margins_path, output_check_path, output_metrics_path, output_stock_check_path,
            today, cur_ts))

        # run dynamic pricing
        args = [
            app_path,
            "--cluster", self.cluster,
            "--config-table", self.config_path,
            "--input-erp-table", self.input_erp_table,
            "--input-demand-table", self.input_demand_table,
            "--output-prices-path", output_prices_path,
            "--output-margins-path", output_margins_path,
            "--output-check-path", output_check_path,
            "--output-metrics-path", output_metrics_path,
            "--output-stock-check-path", output_stock_check_path,
            "--date", today,
            "--timestamp", cur_ts,
        ]

        env = os.environ.copy()
        env.update({
            "YT_TOKEN": self.Parameters.oauth_token.data(),
        })

        with sdk2.helpers.ProcessLog(self, logger="dynamic_pricing") as pl:
            sdk2.helpers.subprocess.check_call(
                args, env=env, stdout=pl.stdout, stderr=pl.stderr, close_fds=True
            )

        # if we are here then no exception was thrown and autostrategy is done
        self._send_step_event(
            path=output_prices_path + "/" + today,
            cur_ts=cur_ts
        )

    @property
    def environment(self):
        return self.Parameters.event_params.get('environment', self.Parameters.environment)

    @property
    def build_type(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.build_type
        else:
            return self.Parameters.event_params.get('build_type', self.Parameters.build_type)

    @property
    def base_yt_dir(self):
        return '//home/market/{env}/monetize/dynamic_pricing'.format(env=self.environment)

    @property
    def cur_ts(self):
        ts = self.Parameters.event_params.get('timestamp', None)
        if ts:
            return datetime.datetime.strptime(ts,'%Y-%m-%dT%H:%M:%S')
        else:
            return datetime.datetime.now()

    @property
    def cluster(self):
        return self.Parameters.event_params.get('cluster', self.Parameters.cluster)

    @property
    def config_path(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.config_path
        else:
            return self.Parameters.event_params.get('config_path', self.base_yt_dir + '/config/latest')

    @property
    def input_erp_table(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.input_erp_table
        default_value = "{}/autostrategy_stats_per_sku/latest".format(self.yt_dest_directory)
        return self.Parameters.event_params.get('path', default_value)

    @property
    def input_demand_table(self):
        if self.Parameters.input_demand_dir:
            yesterday = self.cur_ts - datetime.timedelta(days=1)
            return self.Parameters.input_demand_dir + "/" + yesterday.strftime("%Y-%m-%d")
        return "{}/elasticity/latest".format(self.yt_dest_directory)

    @property
    def yt_dest_directory(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.yt_dest_directory
        else:
            return self.base_yt_dir

    @property
    def app_type(self):
        if self.build_type == ctt.ReleaseStatus.TESTING:
            return MarketDynamicPricingAppTest
        else:
            return MarketDynamicPricingApp

    def _get_app(self):
        if self.build_type:
            # New pipeline
            rsc = self.app_type.find(state=ctr.State.READY).order(-sdk2.Resource.id).first()
            logging.info("Found {} resource #{} that should contain dynamic pricing application"
                            .format(rsc.type, rsc.id))
        else:
            # For compatibility only
            # (TODO: remove when new pipeline will be released)
            if self.Parameters.use_latest_version:
                rsc = MarketDynamicPricingApp.find(state=ctr.State.READY).order(-sdk2.Resource.id).first()
                logging.info("Found {} resource #{} that should contain dynamic pricing application"
                             .format(rsc.type, rsc.id))
            else:
                rsc = self.Parameters.app_resource
                logging.info("Taken {} resource #{} that should contain dynamic pricing application"
                             .format(rsc.type, rsc.id))

        # unpack package
        package_path = str(sdk2.ResourceData(rsc).path)
        sdk2.helpers.subprocess.Popen("tar -xf {}".format(package_path), shell=True).wait()

        app_path = "./dynamic_pricing_autostrategy"  # should be in the package

        return app_path

    def _send_step_event(self, path, cur_ts):
        if not self.Parameters.step_event:
            return

        logging.info("Create STEP event 'cluster_table_publish'...")
        resp = requests.post(
            url="https://step.sandbox.yandex-team.ru/api/v1/events",
            headers={"Authorization": "OAuth {}".format(self.Parameters.oauth_token.data())},
            json={"events": [{
                "name": "cluster_table_publish",
                "params": {
                    "cluster": self.cluster,
                    "path": path,
                    "type": "market-dynamic-pricing-log",
                    "group": "dynamic_pricing_autostrategy_done",
                    "scale": "1d",
                    "timestamp": cur_ts,
                    "environment": self.environment,
                    "build_type": self.build_type
                }
            }]}
        )
        logging.info("STEP response: {}".format(resp.text))
