# -*- coding: utf-8 -*-
import logging
import datetime
import requests

from sandbox import sdk2
from sandbox.projects.common.binary_task import deprecated as binary_task

CUSTOM_ENVIRONMENT = "custom"

class MarketAutostrategyUpdateCurrentPrices(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """Autostrategy Update Current Prices
    """

    class Parameters(sdk2.Task.Parameters):
        oauth_token = sdk2.parameters.Vault("Vault secret contains OAuth token; should provide access to YT", required=True)
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.RadioGroup("Environment for task") as environment:
            environment.values.custom = environment.Value(CUSTOM_ENVIRONMENT, default=True)
            environment.values.testing = environment.Value("testing")
            environment.values.production = environment.Value("production")
        with sdk2.parameters.Group('Notification parameters') as notify_parameters:
            step_event = sdk2.parameters.Bool("Send STEP event", default=False)
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            cluster = sdk2.parameters.String("YT cluster", required=True, default="hahn")
            with environment.value[CUSTOM_ENVIRONMENT]:
                raw_path = sdk2.parameters.String("raw path in YT", required=True)
                filtered_ax_data_path = sdk2.parameters.String("filtered axapta data path in YT", required=True)
                last_curr_price_dir = sdk2.parameters.String("last curr price dir in YT, result'll be here", required=True)

        with sdk2.parameters.Group('STEP parameters') as step_parameters:
            event_params = sdk2.parameters.Dict("Params from STEP event, overrides all")

    def on_execute(self):
        from market.dynamic_pricing.deprectated.autostrategy_update_current_prices.lib import update_current_prices
        timestamp=self.cur_ts
        cur_ts=timestamp.strftime('%Y-%m-%dT%H:%M:%S')
        today=timestamp.strftime("%Y-%m-%d")

        update_current_prices.update_current_prices(
            yt_token=self.Parameters.oauth_token.data(),
            cluster=self.cluster,
            raw_path=self.raw_path,
            filtered_ax_data_path=self.filtered_ax_data_path,
            last_curr_price_dir=self.last_curr_price_dir,
            today=today,
        )
        # if we are here then no exception was thrown and filter is done
        self._send_step_event(
            path=self.last_curr_price_dir + '/latest',
            cur_ts=cur_ts
        )

    @property
    def environment(self):
        return self.Parameters.event_params.get('environment', self.Parameters.environment)
    @property
    def base_yt_dir(self):
        return '//home/market/{env}/monetize/dynamic_pricing'.format(env=self.environment)
    @property
    def cur_ts(self):
        ts=self.Parameters.event_params.get('timestamp', None)
        if  ts:
            return datetime.datetime.strptime(ts,'%Y-%m-%dT%H:%M:%S')
        else:
            return datetime.datetime.now()
    @property
    def cluster(self):
        return self.Parameters.event_params.get('cluster', self.Parameters.cluster)
    @property
    def raw_path(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.raw_path
        else:
            return self.base_yt_dir + '/autostrategy_stats_raw/latest'
    @property
    def filtered_ax_data_path(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.filtered_ax_data_path
        else:
            return self.base_yt_dir + '/filter_data_for_axapta/latest'
    @property
    def last_curr_price_dir(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.last_curr_price_dir
        else:
            return self.base_yt_dir + '/autostrategy_current_prices/sku_price'


    def _send_step_event(self, path, cur_ts):
        if not self.Parameters.step_event:
            return

        logging.info("Create STEP event 'cluster_table_publish'...")
        resp = requests.post(
            url="https://step.sandbox.yandex-team.ru/api/v1/events",
            headers={"Authorization": "OAuth {}".format(self.Parameters.oauth_token.data())},
            json={"events": [{
                "name": "cluster_table_publish",
                "params": {
                    "cluster": self.cluster,
                    "path": path,
                    "type": "market-dynamic-pricing-log",
                    "group": "msku_statistics_autostrategy_current_prices_done",
                    "scale": "1d",
                    "timestamp":  cur_ts,
                    "environment": self.environment,
                }
            }]}
        )
        logging.info("STEP response: {}".format(resp.text))
