# -*- coding: utf-8 -*-
import logging
import datetime
import requests

from sandbox import sdk2
from sandbox.projects.common.binary_task import deprecated as binary_task
import sandbox.common.types.task as ctt

CUSTOM_ENVIRONMENT = "custom"

class MarketCalculateBoundsForAutostrategy(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """Prepare Autostrategy Stats
    """

    class Parameters(sdk2.Task.Parameters):
        oauth_token = sdk2.parameters.Vault("Vault secret contains OAuth token; should provide access to YT", required=True)
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.RadioGroup("Environment for task") as environment:
            environment.values.custom = environment.Value(CUSTOM_ENVIRONMENT, default=True)
            environment.values.testing = environment.Value("testing")
            environment.values.production = environment.Value("production")
        with sdk2.parameters.Group('Notification parameters') as notify_parameters:
            step_event = sdk2.parameters.Bool("Send STEP event", default=False)
        with sdk2.parameters.Group('Build type parameters') as build_type_parameters:
            build_type = sdk2.parameters.String("Build type", required=True, default=ctt.ReleaseStatus.STABLE)
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            cluster = sdk2.parameters.String("YT cluster", required=True, default="hahn")
            with environment.value[CUSTOM_ENVIRONMENT]:
                raw_path = sdk2.parameters.String("raw path in YT", required=True)
                result_dir = sdk2.parameters.String("result dir in YT", required=True)

        with sdk2.parameters.Group('STEP parameters') as step_parameters:
            event_params = sdk2.parameters.Dict("Params from STEP event, overrides all")

    def on_execute(self):
        from market.dynamic_pricing.deprectated.autostrategy_bounds.lib import get_bounds
        timestamp = self.cur_ts
        cur_ts = timestamp.strftime('%Y-%m-%dT%H:%M:%S')
        result_path = self.result_dir + '/' + cur_ts

        get_bounds.get_bounds(
            yt_token=self.Parameters.oauth_token.data(),
            cluster=self.cluster,
            raw_path=self.raw_path,
            result_path=result_path,
        )
        # if we are here then no exception was thrown and filter is done
        self._send_step_event(
            path=result_path,
            cur_ts=cur_ts
        )

    @property
    def environment(self):
        return self.Parameters.event_params.get('environment', self.Parameters.environment)

    @property
    def base_yt_dir(self):
        return '//home/market/{env}/monetize/dynamic_pricing'.format(env=self.environment)

    @property
    def cur_ts(self):
        ts=self.Parameters.event_params.get('timestamp', None)
        if  ts:
            return datetime.datetime.strptime(ts,'%Y-%m-%dT%H:%M:%S')
        else:
            return datetime.datetime.now()

    @property
    def build_type(self):
        return self.Parameters.event_params.get('build_type', self.Parameters.build_type)

    @property
    def cluster(self):
        return self.Parameters.event_params.get('cluster', self.Parameters.cluster)

    @property
    def raw_path(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.raw_path
        else:
            return self.base_yt_dir + '/autostrategy_stats_raw/latest'

    @property
    def result_dir(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.result_dir
        else:
            return self.base_yt_dir + '/autostrategy_stats_per_sku'

    @property
    def config_path(self):
        return self.Parameters.event_params.get('config_path', self.base_yt_dir + '/config/latest')


    def _send_step_event(self, path, cur_ts):
        if not self.Parameters.step_event:
            return

        logging.info("Create STEP event 'cluster_table_publish'...")
        resp = requests.post(
            url="https://step.sandbox.yandex-team.ru/api/v1/events",
            headers={"Authorization": "OAuth {}".format(self.Parameters.oauth_token.data())},
            json={"events": [{
                "name": "cluster_table_publish",
                "params": {
                    "cluster": self.cluster,
                    "path": path,
                    "type": "market-dynamic-pricing-log",
                    "group": "msku_statistics_autostrategy_bounds_done",
                    "scale": "1d",
                    "timestamp":  cur_ts,
                    "config_path": self.config_path,
                    "environment": self.environment,
                    "build_type": self.build_type
                }
            }]}
        )
        logging.info("STEP response: {}".format(resp.text))
