# -*- coding: utf-8 -*-
import logging
import datetime
import requests

from sandbox import sdk2
from sandbox.projects.common.binary_task import deprecated as binary_task
from sandbox.common.telegram import TelegramBot
from sandbox.common.types import task as ctt


class MarketCollectPricesFromReport(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """Collect prices from Market's report for Dynamic Pricing
    """

    class Parameters(sdk2.Task.Parameters):
        oauth_token = sdk2.parameters.Vault("Vault secret contains OAuth token; should provide access to YT, STEP", required=True)
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            cluster = sdk2.parameters.String("YT cluster", required=True, default="hahn")
            group_table = sdk2.parameters.String("Path to input in YT", required=True)
            price_path = sdk2.parameters.String("Path to output in YT", required=True)

        with sdk2.parameters.Group('Report parameters') as report_parameters:
            balancer_host = sdk2.parameters.String("Report balancer host", required=True, default="dynamic-pricing-report.blue.vs.market.yandex.net")
            balancer_port = sdk2.parameters.Integer("Report balancer port", required=True, default=80)
            balancer_threads = sdk2.parameters.Integer("Threads count for collecting", required=True, default=10)

        with sdk2.parameters.Group('Notification parameters') as notify_parameters:
            step_event = sdk2.parameters.Bool("Send STEP event", default=False)
            telegram_notification = sdk2.parameters.Bool("Send report to Telegram", default=False)
            with telegram_notification.value[True]:
                telegram_send_on_success = sdk2.parameters.Bool("Send report on SUCCESS", default=False)
                telegram_chat_id = sdk2.parameters.Integer("Telegram chat ID", required=True)
                telegram_bot_token_vault = sdk2.parameters.Vault("Vault secret contains Telegram bot token", required=True)

    def on_execute(self):
        from market.dynamic_pricing.price_collector.lib import collect
        ts = datetime.datetime.now().strftime('%Y-%m-%dT%H:%M:%S')
        price_table = collect.collect(
            yt_token=self.Parameters.oauth_token.data(),
            cluster=self.Parameters.cluster,
            input_table=self.Parameters.group_table,
            output_path=self.Parameters.price_path,
            output_table_name=ts,
            report_host=self.Parameters.balancer_host,
            report_port=self.Parameters.balancer_port,
            report_threads=self.Parameters.balancer_threads
        )
        self._send_step_event(ts, price_table)

    def _send_step_event(self, ts, price_table):
        if not self.Parameters.step_event:
            return

        logging.info("Create STEP event 'cluster_table_publish'...")
        resp = requests.post(
            url="https://step.sandbox.yandex-team.ru/api/v1/events",
            headers={"Authorization": "OAuth {}".format(self.Parameters.oauth_token.data())},
            json={"events": [{
                "name": "cluster_table_publish",
                "params": {
                    "cluster": self.Parameters.cluster,
                    "path": price_table,
                    "type": "market-dynamic-pricing-log",
                    "group": "dynamic_pricing_prices_collecting_done",
                    "scale": "1d",
                    "timestamp": ts,
                }
            }]}
        )
        logging.info("STEP response: {}".format(resp.text))

    def _notify(self, status):
        if not self.Parameters.telegram_notification:
            return
        if not self.Parameters.telegram_send_on_success and status == ctt.Status.SUCCESS:
            return

        link = 'https://sandbox.yandex-team.ru/task/{}/view'.format(self.id)
        message = "Sandbox task: collect prices from Report #{} has finished: {}\nLink: {}".format(
            self.id,
            status,
            link
        )
        try:
            bot = TelegramBot(bot_token=self.Parameters.telegram_bot_token_vault.data())
            bot.send_message(self.Parameters.telegram_chat_id, message)
        except Exception as e:
            logging.warn("Telegram notification failed: {}".format(e.message))

    def on_finish(self, prev_status, status):
        self._notify(status)

    def on_break(self, prev_status, status):
        self._notify(status)
