# -*- coding: utf-8 -*-
import logging
import datetime
import requests

from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.common.binary_task import deprecated as binary_task
import sandbox.projects.sandbox.resources as sb_resources

CUSTOM_ENVIRONMENT = "custom"


class MarketExportAutostrategyPricesToAxapta(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """Export data to Axapta
    """

    class Requirements(sdk2.Requirements):
        container_resource = sdk2.Requirements.container_resource(default=1167372225)

    class Parameters(sdk2.Task.Parameters):
        oauth_token = sdk2.parameters.Vault("Vault secret contains OAuth token; should provide access to YT", required=True)
        axapta_token = sdk2.parameters.Vault("Vault secret contains axapta token; should provide access to AX", required=True)
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.RadioGroup("Environment for task") as environment:
            environment.values.custom = environment.Value(CUSTOM_ENVIRONMENT, default=True)
            environment.values.testing = environment.Value("testing")
            environment.values.production = environment.Value("production")
        with sdk2.parameters.Group('Notification parameters') as notify_parameters:
            step_event = sdk2.parameters.Bool("Send STEP event", default=False)
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            cluster = sdk2.parameters.String("YT cluster", required=True, default="hahn")
            prices_path = sdk2.parameters.String("prices path in YT", required=True)

        with sdk2.parameters.Group('STEP parameters') as step_parameters:
            event_params = sdk2.parameters.Dict("Params from STEP event, overrides all")

    def on_execute(self):
        from market.dynamic_pricing.deprectated.autostrategy_export_data_to_axapta.lib import export_data
        cur_ts = self.cur_ts
        export_data.export_data(
            yt_token=self.Parameters.oauth_token.data(),
            cluster=self.cluster,
            timestamp=cur_ts,
            table_path=self.prices_path,
            axapta_password=self.Parameters.axapta_token.data(),
            test_flg=False if self.environment == "production" else True
        )

        # if we are here then no exception was thrown and filter is done
        self._send_step_event(
            path=prices_path,
            cur_ts=cur_ts
        )

    @property
    def environment(self):
        return self.Parameters.event_params.get('environment', self.Parameters.environment)

    @property
    def base_yt_dir(self):
        return '//home/market/{env}/monetize/dynamic_pricing'.format(env=self.environment)

    @property
    def cur_ts(self):
        return datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')

    @property
    def cluster(self):
        return self.Parameters.event_params.get('cluster', self.Parameters.cluster)

    @property
    def prices_path(self):
        return self.Parameters.event_params.get('prices_path', self.Parameters.prices_path)

    def _send_step_event(self, path, cur_ts):
        if not self.Parameters.step_event:
            return

        logging.info("Create STEP event 'cluster_table_publish'...")
        resp = requests.post(
            url="https://step.sandbox.yandex-team.ru/api/v1/events",
            headers={"Authorization": "OAuth {}".format(self.Parameters.oauth_token.data())},
            json={"events": [{
                "name": "cluster_table_publish",
                "params": {
                    "cluster": self.cluster,
                    "path": path,
                    "type": "market-dynamic-pricing-log",
                    "group": "axapta-dpautostrategy-prices",
                    "scale": "1d",
                    "timestamp":  cur_ts,
                    "environment": self.environment,
                    "test_flg": False if self.environment == "production" else True
                }
            }]}
        )
        logging.info("STEP response: {}".format(resp.text))
