# -*- coding: utf-8 -*-
import logging
import datetime
import requests

from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.common.binary_task import deprecated as binary_task

CUSTOM_ENVIRONMENT = "custom"

class MarketFilterAutostrategyPricesForAxapta(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """Filter data to load into Axapta
    """

    class Parameters(sdk2.Task.Parameters):
        oauth_token = sdk2.parameters.Vault("Vault secret contains OAuth token; should provide access to YT", required=True)
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.RadioGroup("Environment for task") as environment:
            environment.values.custom = environment.Value(CUSTOM_ENVIRONMENT, default=True)
            environment.values.testing = environment.Value("testing")
            environment.values.production = environment.Value("production")
        with sdk2.parameters.Group('Notification parameters') as notify_parameters:
            step_event = sdk2.parameters.Bool("Send STEP event", default=False)
            send_full = sdk2.parameters.Bool("Send full or diff to axapta", default=False)
        with sdk2.parameters.Group('Build type parameters') as build_type_parameters:
            build_type = sdk2.parameters.String("Build type", required=True, default=ctt.ReleaseStatus.STABLE)
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            cluster = sdk2.parameters.String("YT cluster", required=True, default="hahn")
            with environment.value[CUSTOM_ENVIRONMENT]:
                full_result_dir = sdk2.parameters.String("full result dir in YT", required=True)
                diff_result_dir = sdk2.parameters.String("diff result dir in YT", required=True)
                prices_path = sdk2.parameters.String("prices path in YT", required=True)
                check_path = sdk2.parameters.String("check path in YT", required=True)
                exp_plan_path = sdk2.parameters.String("experiment plan path in YT", required=True)
                kvi_path = sdk2.parameters.String("kvi path in YT", required=True)
                prev_full_dir = sdk2.parameters.String("dir with previous full", required=True)

        with sdk2.parameters.Group('STEP parameters') as step_parameters:
            event_params = sdk2.parameters.Dict("Params from STEP event, overrides all")

    def on_execute(self):
        from market.dynamic_pricing.deprectated.autostrategy_filter_data_for_axapta.lib import filter_data
        timestamp = self.cur_ts
        cur_ts = timestamp.strftime('%Y-%m-%dT%H:%M:%S')
        result_path = filter_data.filter_data(
            yt_token=self.Parameters.oauth_token.data(),
            cluster=self.cluster,
            send_full=self.send_full,
            kvi_path=self.kvi_path,
            exp_plan_path=self.exp_plan_path,
            prices_path=self.prices_path,
            check_path=self.check_path,
            prev_full_dir=self.prev_full_dir,
            full_result_path=self.full_result_dir + '/' + cur_ts,
            diff_result_path=self.diff_result_dir + '/' + cur_ts,
            timestamp=cur_ts
        )

        # if we are here then no exception was thrown and filter is done
        self._send_step_event(
            path=result_path,
            cur_ts=cur_ts
        )

    @property
    def environment(self):
        return self.Parameters.event_params.get('environment', self.Parameters.environment)

    @property
    def base_yt_dir(self):
        return '//home/market/{env}/monetize/dynamic_pricing'.format(env=self.environment)

    @property
    def cur_ts(self):
        ts = self.Parameters.event_params.get('timestamp', None)
        if ts:
            return datetime.datetime.strptime(ts, '%Y-%m-%dT%H:%M:%S')
        else:
            return datetime.datetime.now()

    @property
    def cluster(self):
        return self.Parameters.event_params.get('cluster', self.Parameters.cluster)

    @property
    def build_type(self):
        return self.Parameters.event_params.get('build_type', self.Parameters.build_type)

    @property
    def send_full(self):
        return self.Parameters.event_params.get('send_full', self.Parameters.send_full)

    @property
    def full_result_dir(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.full_result_dir
        else:
            return self.base_yt_dir + '/filter_data_for_axapta'

    @property
    def diff_result_dir(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.diff_result_dir
        else:
            return self.base_yt_dir + '/diff_filter_data_for_axapta'

    @property
    def prev_full_dir(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.prev_full_dir
        else:
            return self.base_yt_dir + '/filter_data_for_axapta'

    @property
    def prices_path(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.prices_path
        else:
            return self.base_yt_dir + '/output/prices/latest'

    @property
    def kvi_path(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.kvi_path
        else:
            return self.base_yt_dir + '/kvi/latest'

    @property
    def check_path(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.check_path
        else:
            return self.base_yt_dir + '/output/check/latest'

    @property
    def exp_plan_path(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.exp_plan_path
        else:
            return self.base_yt_dir + '/experiment_plan/latest'

    def _send_step_event(self, path, cur_ts):
        if not self.Parameters.step_event:
            return

        logging.info("Create STEP event 'cluster_table_publish'...")
        resp = requests.post(
            url="https://step.sandbox.yandex-team.ru/api/v1/events",
            headers={"Authorization": "OAuth {}".format(self.Parameters.oauth_token.data())},
            json={"events": [{
                "name": "cluster_table_publish",
                "params": {
                    "cluster": self.cluster,
                    "path": path,
                    "type": "market-dynamic-pricing-log",
                    "group": "filtered_data_for_axapta",
                    "scale": "1d",
                    "timestamp": cur_ts,
                    "environment": self.environment,
                    "test_flg": False if self.environment == "production" else True,
                    "build_type": self.build_type
                }
            }]}
        )
        logging.info("STEP response: {}".format(resp.text))
