# -*- coding: utf-8 -*-
import logging
import datetime
import requests

from sandbox import sdk2
from sandbox.projects.common.binary_task import deprecated as binary_task
import sandbox.common.types.task as ctt

CUSTOM_ENVIRONMENT = "custom"

class MarketPrepareRawForAutostrategyStats(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """Prepare Autostrategy Stats
    """

    class Parameters(sdk2.Task.Parameters):
        oauth_token = sdk2.parameters.Vault("Vault secret contains OAuth token; should provide access to YT", required=True)
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.RadioGroup("Environment for task") as environment:
            environment.values.custom = environment.Value(CUSTOM_ENVIRONMENT, default=True)
            environment.values.testing = environment.Value("testing")
            environment.values.production = environment.Value("production")
        with sdk2.parameters.Group('Notification parameters') as notify_parameters:
            step_event = sdk2.parameters.Bool("Send STEP event", default=False)
        with sdk2.parameters.Group('Build type parameters') as build_type_parameters:
            build_type = sdk2.parameters.String("Build type", default=ctt.ReleaseStatus.STABLE)
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            cluster = sdk2.parameters.String("YT cluster", required=True, default="hahn")
            hourly_stock_time_limit = sdk2.parameters.Integer("life time for hourly stock, hours", required=True, default=12)
            use_latest = sdk2.parameters.Bool("Use latest tables or by date", default=True)
            canon = sdk2.parameters.Bool("Use canonic paths for input paths", default=False)
            with environment.value[CUSTOM_ENVIRONMENT]:
                config_path = sdk2.parameters.String("config path in YT", required=True)
                sku_list_path = sdk2.parameters.String("sku list path in YT", required=True)
                exp_schedule_path = sdk2.parameters.String("exp schedule path in YT", required=True)
                last_curr_price_path = sdk2.parameters.String("last curr price path in YT", required=True)
                deadstock_sales_path = sdk2.parameters.String("deadstock sales path in YT", required=True)
                result_dir = sdk2.parameters.String("result dir in YT", required=True)

        with sdk2.parameters.Group('STEP parameters') as step_parameters:
            event_params = sdk2.parameters.Dict("Params from STEP event, overrides all")


    def on_execute(self):
        from market.dynamic_pricing.deprectated.autostrategy_stats_raw.lib import get_stats_raw
        from market.dynamic_pricing.deprectated.tests.lib.pipeline_test import YtClient
        from market.dynamic_pricing.deprectated.utilities.lib.yt_paths import AutostrategyPipelinePaths as paths_by_env

        timestamp = self.cur_ts
        cur_ts = timestamp.strftime('%Y-%m-%dT%H:%M:%S')
        yt_client = YtClient(
            cluster=self.Parameters.cluster,
            cur_ts=cur_ts,
            yt_token=self.Parameters.oauth_token.data(),
        )

        max_hourly_stock_table_name_ts = timestamp - datetime.timedelta(hours=self.hourly_stock_time_limit)

        params_dict = self.get_parameters_dict(exclude=['env', 'canon'])
        logging.info("params: canon={0} {1} other {1}".format(self.canon, self.Parameters.canon, str(params_dict)))
        paths = paths_by_env(self.environment, canon=False, **params_dict)
        paths_canon = paths_by_env(self.environment, canon=self.canon, **params_dict)
        config_path = yt_client.latest_table_by_path('config', paths_canon.config_path)
        result_path = paths.raw_stats_path + '/' + cur_ts

        get_stats_raw.get_stats_raw(
            yt_token=self.Parameters.oauth_token.data(),
            cluster=self.cluster,
            cur_ts=cur_ts,
            result_path=result_path,
            use_latest=self.use_latest,
            config_path=config_path,
            sku_list_path=yt_client.latest_table_by_path('groups', paths_canon.groups_path),
            exp_schedule_path=yt_client.latest_table_by_path('exp_schedule', paths_canon.exp_schedule_path),
            last_curr_price_path=yt_client.latest_table_by_path('sku_price', paths_canon.sku_price_path),
            deadstock_sales_path=yt_client.latest_table_by_path('deadstock_sales_path', paths_canon.deadstock_sales_path),
            max_hourly_stock_table_name_ts=max_hourly_stock_table_name_ts,
        )
        # if we are here then no exception was thrown and filter is done
        self._send_step_event(
            path=result_path,
            cur_ts=cur_ts,
            config_path=config_path
        )

    @property
    def environment(self):
        return self.Parameters.event_params.get('environment', self.Parameters.environment)

    @property
    def build_type(self):
        return self.Parameters.event_params.get('build_type', self.Parameters.build_type)

    @property
    def use_latest(self):
        return self.Parameters.event_params.get('use_latest', self.Parameters.use_latest)

    @property
    def canon(self):
        return self.Parameters.event_params.get('canon', self.Parameters.canon)

    @property
    def base_yt_dir(self):
        return '//home/market/{env}/monetize/dynamic_pricing'.format(env=self.environment)

    @property
    def cur_ts(self):
        ts = self.Parameters.event_params.get('timestamp', None)
        if ts:
            return datetime.datetime.strptime(ts, '%Y-%m-%dT%H:%M:%S')
        else:
            return datetime.datetime.now()

    @property
    def hourly_stock_time_limit(self):
        return self.Parameters.event_params.get('hourly_stock_time_limit', self.Parameters.hourly_stock_time_limit)

    @property
    def cluster(self):
        return self.Parameters.event_params.get('cluster', self.Parameters.cluster)

    def get_parameters_dict(self, exclude):
        return {
            k:getattr(self.Parameters, k)
            for k in dir(self.Parameters)
            if not k.startswith('__') and not callable(k) and k not in exclude
        }

    def _send_step_event(self, path, cur_ts, config_path):
        if not self.Parameters.step_event:
            return

        logging.info("Create STEP event 'cluster_table_publish'...")
        resp = requests.post(
            url="https://step.sandbox.yandex-team.ru/api/v1/events",
            headers={"Authorization": "OAuth {}".format(self.Parameters.oauth_token.data())},
            json={"events": [{
                "name": "cluster_table_publish",
                "params": {
                    "cluster": self.cluster,
                    "path": path,
                    "type": "market-dynamic-pricing-log",
                    "group": "msku_statistics_autostrategy_raw_done",
                    "scale": "1d",
                    "timestamp": cur_ts,
                    "config_path": config_path,
                    "environment": self.environment,
                    "build_type": self.build_type
                }
            }]}
        )
        logging.info("STEP response: {}".format(resp.text))
