# -*- coding: utf-8 -*-
import logging
import datetime
import requests
import json

from sandbox import sdk2
from sandbox.projects.common.binary_task import deprecated as binary_task

CUSTOM_ENVIRONMENT = "custom"

class MarketStoreOutputOfAutostrategyPipelineTest(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """Store Canonic Output Of Autostrategy Pipeline Test"""

    class Parameters(sdk2.Task.Parameters):
        oauth_token = sdk2.parameters.Vault("Vault secret contains OAuth token; should provide access to YT", required=True)
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.RadioGroup("Environment for task") as environment:
            environment.values.custom = environment.Value(CUSTOM_ENVIRONMENT, default=True)
            environment.values.testing = environment.Value("testing")
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            cluster = sdk2.parameters.String("YT cluster", required=True, default="hahn")
            with environment.value[CUSTOM_ENVIRONMENT]:
                source_path = sdk2.parameters.String("Path in YT to take the canonic results from", required=True)
                destination_path = sdk2.parameters.String("Path in YT to put the canonic results into", required=True)
                results_suffixes_json = sdk2.parameters.String("Json with locations of results in source_path", required=True)
        with sdk2.parameters.Group('Runtime parameters') as runtime_parameters:
            timestamp = sdk2.parameters.String("Canonic timestamp", required=True)

    def on_execute(self):
        from market.dynamic_pricing.deprectated.tests.lib.results_collector import results_collector
        cur_ts=self.cur_ts.strftime('%Y-%m-%dT%H:%M:%S')

        results_collector(
            yt_token=self.Parameters.oauth_token.data(),
            cluster=self.cluster,
            test_output_dir=self.test_output,
            canon_dir=self.canon_output,
            results_paths_dict=self.result_paths,
            cur_ts=cur_ts,
        )


    @property
    def cluster(self):
        return self.Parameters.cluster

    @property
    def environment(self):
        return self.Parameters.environment

    @property
    def base_yt_dir(self):
        return '//home/market/{env}/monetize/dynamic_pricing'.format(env=self.environment)

    @property
    def cur_ts(self):
        return datetime.datetime.strptime(self.Parameters.timestamp, '%Y-%m-%dT%H:%M:%S')

    @property
    def test_output(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.source_path
        else:
            return self.base_yt_dir

    @property
    def canon_output(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.destination_path
        else:
            return self.base_yt_dir + '/canonic_test'

    @property
    def result_paths(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return json.loads(self.Parameters.results_suffixes_json)
        else:
            return {
                "stats_raw": "autostrategy_stats_raw",
                "bounds": "autostrategy_stats_per_sku",
                "output_check": "output/check",
                "output_margins": "output/margins",
                "output_metrics": "output/metrics",
                "output_prices": "output/prices",
                "output_stock_check": "output/stock_check",
                "filter_data": "filter_data_for_axapta"
            }
