import os
import logging
import datetime
import requests

from sandbox import sdk2
from sandbox.projects.common.binary_task import deprecated as binary_task
from sandbox.common.types import task as ctt

CUSTOM_ENVIRONMENT = "custom"


class RunMarketAutostrategyPipelineTest(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """Run autostrategy pipeline tests and wait for results"""

    class Parameters(sdk2.Task.Parameters):
        oauth_token = sdk2.parameters.Vault("Vault secret contains OAuth token; should provide access to YT", required=True)
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.RadioGroup("Environment for task") as environment:
            environment.values.custom = environment.Value(CUSTOM_ENVIRONMENT, default=True)
            environment.values.testing = environment.Value("testing")
            environment.values.production = environment.Value("production")
        with sdk2.parameters.Group('Notification parameters') as notify_parameters:
            step_event = sdk2.parameters.Bool("Send STEP event", default=False)
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            cluster = sdk2.parameters.String("YT cluster", required=True, default="hahn")
            with environment.value[CUSTOM_ENVIRONMENT]:
                sku_bounds_path = sdk2.parameters.String("Bounds path in YT", required=True)
                raw_stats_path = sdk2.parameters.String("Stats path in YT", required=True)
                sku_price_path = sdk2.parameters.String("Sku price path in YT", required=True)
                groups_path = sdk2.parameters.String("Groups path in YT", required=True)
                elasticity_path = sdk2.parameters.String("Elasticity path in YT", required=True)
                prices_path = sdk2.parameters.String("New prices table path in YT", required=True)
                margins_path = sdk2.parameters.String("Margins table path in YT", required=True)
                check_path = sdk2.parameters.String("Price check results table path in YT", required=True)
                metrics_path = sdk2.parameters.String("Optimizer metrics results table path in YT", required=True)
                stock_check_path = sdk2.parameters.String("Stock controller check results table path in YT", required=True)
                result_path = sdk2.parameters.String("Filtered results for AXAPTA table path in YT", required=True)
                config_path = sdk2.parameters.String("Config table path in YT", required=True)
                exp_schedule_path = sdk2.parameters.String("Experiment plan table path in YT", required=True)
                expected_result_path = sdk2.parameters.String("Expected test results table path in YT", required=True)
                diff_path = sdk2.parameters.String("Table to store the difference between the canonic output and test output: path in YT", required=True)
        with sdk2.parameters.Group('Runtime parameters') as runtime_parameters:
            interval = sdk2.parameters.Integer("interval", default=5)
            timeout = sdk2.parameters.Integer("timeout", default=3600)
            timestamp = sdk2.parameters.String("Canonic timestamp", required=True)
            mute_comparison = sdk2.parameters.Bool("Mute test vs prod comparison", default=False)
        with sdk2.parameters.Group('Telegram parameters') as telegram_parameters:
            telegram_notification = sdk2.parameters.Bool("Send report to Telegram", default=False)
            with telegram_notification.value[True]:
                telegram_send_on_success = sdk2.parameters.Bool("Send report on SUCCESS", default=False)
                telegram_chat_id = sdk2.parameters.Integer("Telegram chat ID", required=True)
                telegram_bot_token_vault = sdk2.parameters.Vault("Vault secret contains Telegram bot token", required=True)


    def _notify(self, status):
        if not self.Parameters.telegram_notification:
            return
        if not self.Parameters.telegram_send_on_success and status == ctt.Status.SUCCESS:
            return

        link = 'https://sandbox.yandex-team.ru/task/{}/view'.format(self.id)
        message = "Sandbox dynamic pricing release testing task #{} has finished: {}\nLink: {}".format(
            self.id,
            status,
            link
        )
        try:
            bot = TelegramBot(bot_token=self.Parameters.telegram_bot_token_vault.data())
            bot.send_message(self.Parameters.telegram_chat_id, message)
        except Exception as e:
            logging.warn("Telegram notification failed: {}".format(e.message))

    def on_finish(self, prev_status, status):
        self._notify(status)

    def on_break(self, prev_status, status):
        self._notify(status)

    def on_execute(self):
        from market.dynamic_pricing.deprectated.tests.lib.pipeline_test import YtClient
        from market.dynamic_pricing.deprectated.utilities.lib.yt_paths import AutostrategyPipelinePaths as paths_by_env

        cur_ts=self.cur_ts.strftime('%Y-%m-%dT%H:%M:%S')
        yt_client = YtClient(
            cluster=self.Parameters.cluster,
            cur_ts=cur_ts,
            yt_token=self.Parameters.oauth_token.data(),
        )

        params_dict = self.get_parameters_dict(exclude=['env', 'canon'])
        paths = paths_by_env(self.Parameters.environment, canon=False, **params_dict)
        paths_prod = paths_by_env(env="production", canon=False, **params_dict)
        paths_canon = paths_by_env(env="testing", canon=True, **params_dict)

        # Check and update the tables in canon input
        yt_client.update_tables(
            prod_paths={
                'config': paths_prod.config_path,
                'groups': paths_prod.groups_path,
                'sku_price': paths_prod.sku_price_path,
                'elasticity': paths_prod.elasticity_path,
                'exp_schedule': paths_prod.exp_schedule_path,
                'deadstock_sales_path': paths_prod.deadstock_sales_path
            },
            canon_paths={
                'config': paths_canon.config_path,
                'groups': paths_canon.groups_path,
                'sku_price': paths_canon.sku_price_path,
                'elasticity': paths_canon.elasticity_path,
                'exp_schedule': paths_canon.exp_schedule_path,
                'deadstock_sales_path': paths_canon.deadstock_sales_path
            },
            additional_paths={
                'exp_schedule': paths.exp_schedule_path,
            },
        )

        # remove previous results
        yt_client.cleanup(paths.results_paths)
        # start new test
        self._send_step_event(
            canon=True,
            cur_ts=cur_ts
        )
        # wait for result or throw timeout exception
        yt_client.check_tables_exist(
            paths.results_paths,
            interval=self.Parameters.interval,
            timeout=self.Parameters.timeout
        )
        logging.info("All necessary tables exist")

        # if all tables exists compare prices tables (prod vs test)
        if not self.Parameters.mute_comparison:
            yt_client.compare(
                expected_table=paths_canon.output_prices_path + '/' + cur_ts,
                result_table=paths.output_prices_path + '/' + cur_ts,
                diff_path=paths_canon.diff_path # /canonic_test/diff
            )
        # if there is no diff - tests passed successfully
        logging.info("Pipeline tests complete")

    @property
    def cur_ts(self):
        return datetime.datetime.strptime(self.Parameters.timestamp, '%Y-%m-%dT%H:%M:%S')

    @property
    def cluster(self):
        return self.Parameters.event_params.get('cluster', self.Parameters.cluster)

    def get_parameters_dict(self, exclude):
        return {
            k:getattr(self.Parameters, k)
            for k in dir(self.Parameters)
            if not k.startswith('__') and not callable(k) and k not in exclude
        }

    def _send_step_event(self, canon, cur_ts):
        logging.info("Create STEP event 'autostrategy_pipeline_test")
        resp = requests.post(
            url="https://step.sandbox.yandex-team.ru/api/v1/events",
            headers={"Authorization": "OAuth {}".format(self.Parameters.oauth_token.data())},
            json={"events": [{
                "name": "cluster_table_publish",
                "params": {
                    "cluster": self.Parameters.cluster,
                    "canon": canon,
                    "type": "market-dynamic-pricing-log",
                    "group": "autostrategy_pipepine_test_start",
                    "timestamp": cur_ts,
                    "environment": self.Parameters.environment,
                    "build_type": ctt.ReleaseStatus.TESTING
                }
            }]}
        )
        logging.info("STEP response: {}".format(resp.text))
