# -*- coding: utf-8 -*-
import logging
import datetime
import requests

from sandbox import sdk2
from sandbox.projects.common.binary_task import deprecated as binary_task

CUSTOM_ENVIRONMENT = "custom"

class MarketQuickChangeAutostrategyPrices(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """Quick Change Autostrategy Prices
    """

    class Parameters(sdk2.Task.Parameters):
        oauth_token = sdk2.parameters.Vault("Vault secret contains OAuth token; should provide access to YT", required=True)
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.RadioGroup("Environment for task") as environment:
            environment.values.custom = environment.Value(CUSTOM_ENVIRONMENT, default=True)
            environment.values.testing = environment.Value("testing")
            environment.values.production = environment.Value("production")
        with sdk2.parameters.Group('Notification parameters') as notify_parameters:
            step_event = sdk2.parameters.Bool("Send STEP event", default=False)
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            cluster = sdk2.parameters.String("YT cluster", required=True, default="hahn")
            with environment.value[CUSTOM_ENVIRONMENT]:
                groups_path = sdk2.parameters.String("groups path in YT", required=True)
                res_dir = sdk2.parameters.String("dir to table with cancel results in YT; will be send to axa", required=True)

        with sdk2.parameters.Group('User parameters') as user_parameters:
            groups_to_cancel_price = sdk2.parameters.String("groups to cancel price; each group from new line; is high priority then skus", required=False, multiline=True, default=None)
            skus_to_change_price = sdk2.parameters.String("""input json [ {"sku":id, "new_price":price},{"sku":id2, "new_price":price2} ]""", required=False, multiline=True, default=None)

    def on_execute(self):
        from market.dynamic_pricing.deprectated.autostrategy_prices_quick_change.lib import change_prices
        timestamp=self.cur_ts
        cur_ts=timestamp.strftime('%Y-%m-%dT%H:%M:%S')

        change_prices.change_prices(
            yt_token=self.Parameters.oauth_token.data(),
            cluster=self.cluster,
            groups_path=self.groups_path,
            res_path=self.res_dir + '/' + cur_ts,
            groups_to_cancel_price=self.groups_to_cancel_price,
            skus_to_change_price=self.skus_to_change_price,
        )
        # if we are here then no exception was thrown and filter is done
        self._send_step_event(
            path=self.res_dir + '/latest',
            cur_ts=cur_ts
        )

    @property
    def environment(self):
        return self.Parameters.environment
    @property
    def base_yt_dir(self):
        return '//home/market/{env}/monetize/dynamic_pricing'.format(env=self.environment)
    @property
    def cur_ts(self):
        return datetime.datetime.now()
    @property
    def cluster(self):
        return self.Parameters.cluster

    @property
    def groups_path(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.groups_path
        else:
            return self.base_yt_dir + '/groups/latest'

    @property
    def groups_to_cancel_price(self):
        return self.Parameters.groups_to_cancel_price

    @property
    def skus_to_change_price(self):
        return self.Parameters.skus_to_change_price

    @property
    def res_dir(self):
        if self.environment == CUSTOM_ENVIRONMENT:
            return self.Parameters.res_path
        else:
            return '//home/market/{env}/monetize/dynamic_pricing/autostrategy_prices_quick_change'.format(env=self.environment)


    def _send_step_event(self, path, cur_ts):
        if not self.Parameters.step_event:
            return

        logging.info("Create STEP event 'cluster_table_publish'...")
        resp = requests.post(
            url="https://step.sandbox.yandex-team.ru/api/v1/events",
            headers={"Authorization": "OAuth {}".format(self.Parameters.oauth_token.data())},
            json={"events": [{
                "name": "cluster_table_publish",
                "params": {
                    "cluster": self.cluster,
                    "path": path,
                    "type": "market-dynamic-pricing-log",
                    "group": "filtered_data_for_axapta",
                    "scale": "1d",
                    "timestamp": cur_ts,
                    "environment": self.environment,
                    "test_flg": False if self.environment == "production" else True
                }
            }]}
        )
        logging.info("STEP response: {}".format(resp.text))
