import logging
from sandbox import sdk2
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.common.types.task import Status
from sandbox.common.utils import chain


class DeployKwytTraversalGraph(sdk2.Task):
    """ Builds executables and then runs graph generator for dynamic pricing kwyt traversal graph. """

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600
        nirvana_token_secret = sdk2.parameters.String("Nirvana token secret (sandbox vault - owner:name)", required=True)
        workflow = sdk2.parameters.String("Target Nirvana workflow id", required=False)

        yt_token = sdk2.parameters.String("Yt token secret (Nirvana vault)", required=True)
        yql_token = sdk2.parameters.String("Yql token secret (Nirvana vault)", required=True)
        sandbox_oauth_token = sdk2.parameters.String("Sandbox token (Nirvana vault)", required=True)
        reference_shops_table = sdk2.parameters.String("Reference shops table path", required=False)
        watson_html_result_table = sdk2.parameters.String("Path to watson html collector output table", required=True)
        xpath_table = sdk2.parameters.String("Path to xpath table", required=True)
        sovetnik_selectors_url = sdk2.parameters.String("Sovetnik selectros url", required=True)
        watson_selectors_url = sdk2.parameters.String("Watson selectros url", required=True)
        juggler_host = sdk2.parameters.String("Juggler host for events", required=True)
        juggler_src = sdk2.parameters.String("Juggler source", required=True)
        statface_host = sdk2.parameters.String("Statface host to send events", required=True)
        statface_report = sdk2.parameters.String("Statface report to send events to", required=True)
        stats_dir = sdk2.parameters.String("Dir path for stats", required=True)
        output_table_dir = sdk2.parameters.String("Directory of resulting table", required=True)
        work_dir = sdk2.parameters.String("Work dir for temp files", required=True)
        errors_dir = sdk2.parameters.String("Dir path for error tables", required=True)
        main_cluster = sdk2.parameters.String("Main YT cluster", required=True)
        backup_cluster = sdk2.parameters.String("Backup YT cluster", required=True)
        run_workflow = sdk2.parameters.Bool("Whether to start created workflow instance", required=True)
        target = sdk2.parameters.String("Graph Build Target", required=True)
        env = sdk2.parameters.String("Graph run environment testing/production", required=True)

    def __run_build_task(self, targets, name_of_build, definition_flags='', env_vars=''):
        task = self.server.task({
            "type": YaMakeTask.type,
            "description": "Build " + name_of_build + " for %d (%s)" % (self.id, self.Parameters.description),
            "owner": self.owner,
            "priority": {
                "class": self.Parameters.priority.cls,
                "subclass": self.Parameters.priority.scls,
            },
            "children": True,
            "context": {
                'checkout': True,
                'targets': targets,
                'build_system': 'ya',
                'build_type': 'release',
                'clear_build': True,
                'check_return_code': True,
                'strip_binaries': True,
                'definition_flags': definition_flags,
                'env_vars': env_vars,
            },
        })
        task_id = task["id"]
        self.server.batch.tasks.start.update([task_id])
        return task_id

    def __run_graph_build_task(self, target, options):
        return self.__run_build_task(
            target,
            'dyn_pricing_graph ',
            options,
            "NIRVANA_TOKEN='$(vault:value:"+self.Parameters.nirvana_token_secret+")'")

    def on_execute(self):
        logging.info("Starting graph build for %s", self.Parameters.target)
        options = self.unpack_options()
        if not self.Context.graph_build_task_id:
            self.Context.graph_build_task_id = self.__run_graph_build_task(self.Parameters.target, options)
            raise sdk2.WaitTask([self.Context.graph_build_task_id], set(chain(Status.Group.FINISH, Status.Group.BREAK)), wait_all=True)

    def unpack_options(self):
        options = \
            " -DWORKFLOW_GUID=" + self.Parameters.workflow + \
            " -DYT_TOKEN=" + self.Parameters.yt_token + \
            " -DYQL_TOKEN=" + self.Parameters.yql_token + \
            " -DSANDBOX_OAUTH_TOKEN=" + self.Parameters.sandbox_oauth_token + \
            " -DREFERENCE_SHOPS_TABLE=" + self.Parameters.reference_shops_table + \
            " -DWATSON_HTML_RESULT_TABLE=" + self.Parameters.watson_html_result_table + \
            " -DXPATH_TABLE=" + self.Parameters.xpath_table + \
            " -DSOVETNIK_SELECTORS_URL=" + self.Parameters.sovetnik_selectors_url + \
            " -DWATSON_SELECTORS_URL=" + self.Parameters.watson_selectors_url + \
            " -DJUGGLER_HOST=" + self.Parameters.juggler_host + \
            " -DJUGGLER_SRC=" + self.Parameters.juggler_src + \
            " -DSTATFACE_HOST=" + self.Parameters.statface_host + \
            " -DSTATFACE_REPORT=" + self.Parameters.statface_report + \
            " -DSTATS_DIR=" + self.Parameters.stats_dir + \
            " -DOUTPUT_TABLE_DIR=" + self.Parameters.output_table_dir + \
            " -DWORK_DIR=" + self.Parameters.work_dir + \
            " -DERRORS_DIR=" + self.Parameters.errors_dir + \
            " -DMAIN_CLUSTER=" + self.Parameters.main_cluster + \
            " -DBACKUP_CLUSTER=" + self.Parameters.backup_cluster + \
            " -DRUN_ENV=" + self.Parameters.env + \
            " -DRUN_WORKFLOW=" + str(self.Parameters.run_workflow)
        return options
