import logging
from sandbox import sdk2
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.common.types.task import Status
from sandbox.common.utils import chain


class DeployZoraTraversalGraph(sdk2.Task):
    """ Builds executables and then runs graph generator for dynamic pricing kwyt traversal graph. """
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600
        nirvana_token_secret = sdk2.parameters.String("Nirvana token secret (sandbox vault - owner:name)", required=True)
        workflow = sdk2.parameters.String("Target Nirvana workflow id", required=True)

        sandbox_oauth_token = sdk2.parameters.String("", required=True)
        yt_token = sdk2.parameters.String("", required=True)
        yql_token = sdk2.parameters.String("", required=True)
        mr_account = sdk2.parameters.String("", required=True)
        input_table = sdk2.parameters.String("", required=True)
        output_table = sdk2.parameters.String("", required=True)
        output_dir = sdk2.parameters.String("", required=True)
        timestamp = sdk2.parameters.String("", required=True)
        output_tables_regex = sdk2.parameters.String("", required=True)
        work_dir = sdk2.parameters.String("", required=True)
        zora_source = sdk2.parameters.String("", required=True)
        tvm_secret = sdk2.parameters.String("", required=True)
        tvm_client_id = sdk2.parameters.String("", required=True)
        cluster = sdk2.parameters.String("", required=True)
        backup_cluster = sdk2.parameters.String("", required=True)
        target = sdk2.parameters.String("Graph Build Target", required=True)

    def __run_build_task(self, targets, name_of_build, definition_flags='', env_vars=''):
        task = self.server.task({
            "type": YaMakeTask.type,
            "description": "Build " + name_of_build + " for %d (%s)" % (self.id, self.Parameters.description),
            "owner": self.owner,
            "priority": {
                "class": self.Parameters.priority.cls,
                "subclass": self.Parameters.priority.scls,
            },
            "children": True,
            "context": {
                'checkout': True,
                'targets': targets,
                'build_system': 'ya',
                'build_type': 'release',
                'clear_build': True,
                'check_return_code': True,
                'strip_binaries': True,
                'definition_flags': definition_flags,
                'env_vars': env_vars,
            },
        })
        task_id = task["id"]
        self.server.batch.tasks.start.update([task_id])
        return task_id

    def __run_graph_build_task(self, target, options):
        return self.__run_build_task(
            target,
            'dyn_pricing_graph ',
            options,
            "NIRVANA_TOKEN='$(vault:value:"+self.Parameters.nirvana_token_secret+")'")

    def on_execute(self):
        logging.info("Starting graph build for %s", self.Parameters.target)
        options = self.unpack_options()
        if not self.Context.graph_build_task_id:
            self.Context.graph_build_task_id = self.__run_graph_build_task(self.Parameters.target, options)
            raise sdk2.WaitTask([self.Context.graph_build_task_id], set(chain(Status.Group.FINISH, Status.Group.BREAK)), wait_all=True)

    def unpack_options(self):
        options = \
            " -DWORKFLOW_GUID=" + self.Parameters.workflow + \
            " -DSANDBOX_OAUTH_TOKEN=" + self.Parameters.sandbox_oauth_token + \
            " -DYT_TOKEN=" + self.Parameters.yt_token + \
            " -DYQL_TOKEN=" + self.Parameters.yql_token + \
            " -DMR_ACCOUNT=" + self.Parameters.mr_account + \
            " -DINPUT_TABLE=" + self.Parameters.input_table + \
            " -DOUTPUT_TABLE=" + self.Parameters.output_table + \
            " -DOUTPUT_DIR=" + self.Parameters.output_dir + \
            " -DTIMESTAMP=" + self.Parameters.timestamp + \
            " -DOUTPUT_TABLES_REGEX=" + self.Parameters.output_tables_regex + \
            " -DWORK_DIR=" + self.Parameters.work_dir + \
            " -DZORA_SOURCE=" + self.Parameters.zora_source + \
            " -DTVM_SECRET=" + self.Parameters.tvm_secret + \
            " -DTVM_CLIENT_ID=" + self.Parameters.tvm_client_id + \
            " -DCLUSTER=" + self.Parameters.cluster + \
            " -DBACKUP_CLUSTER=" + self.Parameters.backup_cluster

        return options
