# coding=utf-8

import logging
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.common.telegram import TelegramBot
from sandbox.sandboxsdk import environments


class InconsistentRequestStatusesChecker(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        database_users_map_string = sdk2.parameters.String('Database users: '
                                                           'String in format (warehouseId, userNameVault)'
                                                           '[;(warehouseId, userNameVault)]', required=True)
        database_passwords_map_string = sdk2.parameters.String('Database passwords: '
                                                               'String in format (warehouseId, passwordVault)'
                                                               '[;(warehouseId, passwordVault)]', required=True)
        database_hosts_map_string = sdk2.parameters.String('Database hosts:'
                                                           'String in format (warehouseId, databseHost)'
                                                           '[;(warehouseId, databaseHost)]', required=True)
        database_ports_map_string = sdk2.parameters.String('Database ports:'
                                                           'String in format (warehouseId, databsePort)'
                                                           '[;(warehouseId, databasePort)]', required=False)
        ffwf_database_user_vault_key = sdk2.parameters.String('FFWF databse user vault key',
                                                              required=True)
        ffwf_database_password_vault_key = sdk2.parameters.String('FFWF databse password vault key',
                                                                  required=True)
        telegram_bot_token_vault_key = sdk2.parameters.String('Telegram bot token key', required=True)
        telegram_chat_id = sdk2.parameters.String('Telegram chat id', required=True)
        inbound_status_mappings = sdk2.parameters.String('Inbound status mappings: '
                                                         'String in format (ffwfStatus, wmsStatus)'
                                                         '[;(ffwfStatus, wmsStatus)]', required=True)
        outbound_status_mappings = sdk2.parameters.String('Outbound status mappings: '
                                                          'String in format (ffwfStatus, wmsStatus)'
                                                          '[;(ffwfStatus, wmsStatus)]', required=True)
        inbound_types = sdk2.parameters.String('Inbound types: String in format ffwfType[;ffwfType]', required=True)
        outbound_types = sdk2.parameters.String('Outbound types: String in format ffwfType[;ffwfType]', required=True)
        inbound_types_to_check_trailer = sdk2.parameters.String('Inbound types for which trailer should be checked: '
                                                                'String in format ffwfType[;ffwfType]', required=True)
        inbound_ffwf_status_for_closed_trailer = sdk2.parameters.String('Inbound FFWF status for closed trailer', required=True)
        inbound_wms_status_for_closed_trailer = sdk2.parameters.String('Inbound FFWF status for closed trailer', required=True)
        st_user_agent = sdk2.parameters.String('Startrack user agent', required=True)
        st_url = sdk2.parameters.String('Startrack url', required=True)
        st_token_vault_key = sdk2.parameters.String('Startrack token vault key', required=True)
        st_queue = sdk2.parameters.String('Startrack queue', required=True)
        st_summary = sdk2.parameters.String('Startrack ticket summary', required=True)
        ffwf_statuses_not_for_monitoring = sdk2.parameters.String('FFWF statuses not for monitoring: '
                                                                  'String in format ffwfStatus[,ffwfStatus]',
                                                                  required=True)
        warehouses_for_monitoring = sdk2.parameters.String('Warehouses for monitoring:'
                                                           'String in format warehouseId[,warehouseId]', required=True)
        tag_for_warehouse = sdk2.parameters.String('Tag for warehouse: String in format (warehouseId, tag)'
                                                   '[;(warehouseId, tag)]', required=True)
        board_for_warehouse = sdk2.parameters.String('Board for warehouse: String in format (warehouseId, boardUrl)'
                                                     '[;(warehouseId, boardUrl)]', required=True)

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5
        environments = (environments.PipEnvironment('psycopg2-binary'),
                        environments.PipEnvironment('pymssql==2.1.4'),
                        environments.PipEnvironment('yandex_tracker_client', version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
                        environments.PipEnvironment('startrek_client', version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"]))

    def checkWmsStatus(self, service_id, requests_info):
        conn = None
        cur = None
        try:
            inbound_types = self.get_inbound_types()
            outbound_types = self.get_outbound_types()
            conn = self.create_wms_connection(service_id)

            cur = conn.cursor()
            i = 0
            j = 0
            receipt_keys = ''
            order_keys = ''
            receipt_key_to_status = {}
            order_key_to_status = {}
            for info in requests_info:
                service_request_id = info[1]
                type = info[3]
                if inbound_types.__contains__(type):
                    receipt_keys += "'" + service_request_id + "'"
                    i += 1
                    if i == 10:
                        cur.execute("select r.RECEIPTKEY, r.STATUS, max(rsh.ADDDATE) from scprd.wmwhse1.RECEIPT (nolock) r "
                                    "left join scprd.wmwhse1.RECEIPTSTATUSHISTORY (nolock) rsh on r.RECEIPTKEY = rsh.RECEIPTKEY "
                                    "and r.STATUS = rsh.STATUS "
                                    "where r.RECEIPTKEY in (" + receipt_keys + ") group by r.RECEIPTKEY, r.STATUS")
                        raw_result = cur.fetchall()
                        for row in raw_result:
                            receipt_key_to_status[str(row[0])] = (self.delete_leading_zeros(str(row[1])), str(row[2]))
                        i = 0
                        receipt_keys = ''
                    else:
                        receipt_keys += ', '
                if outbound_types.__contains__(type):
                    order_keys += "'" + service_request_id + "'"
                    j += 1
                    if j == 10:
                        cur.execute("select o.ORDERKEY, o.STATUS, max(osh.ADDDATE) from scprd.wmwhse1.ORDERS (nolock) o "
                                    "left join scprd.wmwhse1.ORDERSTATUSHISTORY (nolock) osh on o.ORDERKEY = osh.ORDERKEY "
                                    "and o.STATUS = osh.STATUS "
                                    "where o.ORDERKEY in (" + order_keys + ") group by o.ORDERKEY, o.STATUS")
                        raw_result = cur.fetchall()
                        for row in raw_result:
                            order_key_to_status[str(row[0])] = (self.delete_leading_zeros(str(row[1])), str(row[2]))
                        j = 0
                        order_keys = ''
                    else:
                        order_keys += ', '
            if i != 0:
                cur.execute("select r.RECEIPTKEY, r.STATUS, max(rsh.ADDDATE) from scprd.wmwhse1.RECEIPT (nolock) r "
                            "left join scprd.wmwhse1.RECEIPTSTATUSHISTORY (nolock) rsh on r.RECEIPTKEY = rsh.RECEIPTKEY "
                            "and r.STATUS = rsh.STATUS "
                            "where r.RECEIPTKEY in (" + receipt_keys[:-2] + ") group by r.RECEIPTKEY, r.STATUS")
                raw_result = cur.fetchall()
                for row in raw_result:
                    receipt_key_to_status[str(row[0])] = (self.delete_leading_zeros(str(row[1])), str(row[2]))
            if j != 0:
                cur.execute("select o.ORDERKEY, o.STATUS, max(osh.ADDDATE) from scprd.wmwhse1.ORDERS (nolock) o "
                            "left join scprd.wmwhse1.ORDERSTATUSHISTORY (nolock) osh on o.ORDERKEY = osh.ORDERKEY "
                            "and o.STATUS = osh.STATUS "
                            "where o.ORDERKEY in (" + order_keys[:-2] + ") group by o.ORDERKEY, o.STATUS")
                raw_result = cur.fetchall()
                for row in raw_result:
                    order_key_to_status[str(row[0])] = (self.delete_leading_zeros(str(row[1])), str(row[2]))

            return receipt_key_to_status, order_key_to_status
        except Exception as error:
            logging.info(error)
        finally:
            if cur is not None:
                cur.close()
            if conn is not None:
                conn.close()
            logging.info('WMS database connection closed.')

    def ffwfRequests(self):
        conn = None
        cur = None
        try:
            user = sdk2.Vault.data(self.Parameters.ffwf_database_user_vault_key)
            password = sdk2.Vault.data(self.Parameters.ffwf_database_password_vault_key)
            logging.info('Connecting to the FFWF database...')
            import psycopg2
            conn = psycopg2.connect(
                host="market-ff-workflow01f.db.yandex.net",
                port="6432",
                database="market_ff_workflow",
                user=user,
                password=password)
            logging.info('Connected')

            cur = conn.cursor()

            cur.execute("select sr.id, sr.service_id, sr.service_request_id, sr.status, sr.type, sr.subtype,"
                        "max(rsh.created_at) from shop_request sr "
                        "left join request_status_history rsh on sr.id = rsh.request_id and sr.status = rsh.status "
                        "where sr.status not in (" + str(self.Parameters.ffwf_statuses_not_for_monitoring) + ") "
                        "and sr.service_id in (" + str(self.Parameters.warehouses_for_monitoring) + ") "
                        "and sr.service_request_id is not null "
                        "and coalesce(sr.only_internal, false) = false "
                        "group by sr.id, sr.service_id, sr.service_request_id, sr.status, sr.type, sr.subtype")
            raw_result = cur.fetchall()

            service_to_info = {}
            for row in raw_result:
                ffwf_id = row[0]
                service_id = row[1]
                service_request_id = row[2]
                status = row[3]
                type = row[4]
                subtype = row[5]
                date = row[6]
                if not service_to_info.__contains__(service_id):
                    service_to_info[service_id] = []
                service_to_info[service_id].append((str(ffwf_id), str(service_request_id), str(status), str(type), str(subtype), str(date)))

            cur.close()
            conn.close()

            return service_to_info
        except Exception as error:
            logging.info(error)
        finally:
            if cur is not None:
                cur.close()
            if conn is not None:
                conn.close()
            logging.info('FFWF database connection closed.')

    def inbounds_having_closed_trailers(self, ffwf_requests_info, wms_requests_statuses, service_id):
        inbound_types_to_check_trailer = self.get_inbound_types_to_check_trailer()
        wms_receipt_statuses = wms_requests_statuses[0]
        inbounds_to_check_trailers = []
        for info in ffwf_requests_info:
            service_request_id = info[1]
            ffwf_status = info[2]
            type = info[3]
            if inbound_types_to_check_trailer.__contains__(type) and ffwf_status == self.Parameters.inbound_ffwf_status_for_closed_trailer\
                and wms_receipt_statuses.__contains__(service_request_id):
                wms_status = wms_receipt_statuses[service_request_id][0]
                if wms_status == self.Parameters.inbound_wms_status_for_closed_trailer:
                    inbounds_to_check_trailers.append(service_request_id)
        inbounds_having_closed_trailers = []
        conn = None
        cur = None
        try:
            conn = self.create_wms_connection(service_id)
            cur = conn.cursor()
            i = 0
            receipt_keys = ''
            for service_request_id in inbounds_to_check_trailers:
                receipt_keys += "'" + service_request_id + "'"
                i += 1
                if i == 10:
                    cur.execute("SELECT rc.receiptKey  FROM scprd.wmwhse1.receipt rc "
                                "JOIN scprd.wmwhse1.trailer tr ON rc.trailerKey = tr.trailerKey "
                                "JOIN scprd.wmwhse1.TRAILERSTATUS ts ON rc.trailerKey = ts.trailerKey "
                                "WHERE rc.receiptKey IN (" + receipt_keys + ") AND ts.TRAILERSTATUS = '5COMP'")
                    raw_result = cur.fetchall()
                    for row in raw_result:
                        inbounds_having_closed_trailers.append(row[0])
                    i = 0
                    receipt_keys = ''
                else:
                    receipt_keys += ', '
            if i != 0:
                cur.execute("SELECT rc.receiptKey  FROM scprd.wmwhse1.receipt rc "
                            "JOIN scprd.wmwhse1.trailer tr ON rc.trailerKey = tr.trailerKey "
                            "JOIN scprd.wmwhse1.TRAILERSTATUS ts ON rc.trailerKey = ts.trailerKey "
                            "WHERE rc.receiptKey IN (" + receipt_keys[:-2] + ") AND ts.TRAILERSTATUS = '5COMP'")
                raw_result = cur.fetchall()
                for row in raw_result:
                    inbounds_having_closed_trailers.append(row[0])
            return inbounds_having_closed_trailers
        except Exception as error:
            logging.info(error)
        finally:
            if cur is not None:
                cur.close()
            if conn is not None:
                conn.close()
            logging.info('WMS database connection closed.')

    def compare_statuses(self, ffwf_requests_info, wms_requests_statuses, inbounds_having_closed_trailers):
        wms_receipt_statuses = wms_requests_statuses[0]
        wms_order_statuses = wms_requests_statuses[1]
        inbound_types = self.get_inbound_types()
        outbound_types = self.get_outbound_types()
        ffwf_inbound_status_to_wms_statuses = self.get_ffwf_inbound_status_to_wms_status()
        ffwf_outbound_status_to_wms_statuses = self.get_ffwf_outbound_status_to_wms_status()
        problems_map = {}
        for info in ffwf_requests_info:
            ffwf_id = info[0]
            service_request_id = info[1]
            ffwf_status = info[2]
            type = info[3]
            subtype = info[4]
            ffwf_date = info[5]
            if ffwf_date == 'None':
                ffwf_date = '-'
            if inbound_types.__contains__(type):
                if not wms_receipt_statuses.__contains__(service_request_id):
                    problems_map[ffwf_id] = ('Поставка (' + str(type) + '/' + str(subtype) + '): ' + ffwf_id + '/' + service_request_id + '. FF:  ' +
                                             ffwf_status + ' (' + ffwf_date + ') WMS: не найдена\n\n')
                else:
                    wms_status = wms_receipt_statuses[service_request_id][0]
                    if not ffwf_inbound_status_to_wms_statuses.__contains__(ffwf_status):
                        continue
                    correct_wms_statuses = ffwf_inbound_status_to_wms_statuses[ffwf_status]
                    is_correct = False
                    for correct_wms_status in correct_wms_statuses:
                        if wms_status == correct_wms_status:
                            is_correct = True
                    if not is_correct:
                        if inbounds_having_closed_trailers.__contains__(service_request_id):
                            continue
                        wms_date_string = wms_receipt_statuses[service_request_id][1]
                        parsed_date = None
                        if wms_date_string == 'None':
                            wms_date_string = '-'
                        else:
                            parsed_date = self.parse_date(wms_date_string)
                        now = datetime.now()
                        if parsed_date is None or parsed_date + timedelta(hours=7) < now:
                            problems_map[ffwf_id] = ('Поставка (' + str(type) + '/' + str(subtype) + '): ' + ffwf_id + '/' + service_request_id + '. FF: ' +
                                                     ffwf_status + ' (' + ffwf_date + ') WMS: ' + wms_status + ' (' + wms_date_string + ')\n\n')
            if outbound_types.__contains__(type):
                if not wms_order_statuses.__contains__(service_request_id):
                    problems_map[ffwf_id] = ('Изъятие (' + str(type) + '/' + str(subtype) + '): ' + ffwf_id + '/' + service_request_id + '. FF: ' +
                                             ffwf_status + ' (' + ffwf_date + ') WMS: не найдено\n\n')
                else:
                    wms_status = wms_order_statuses[service_request_id][0]
                    if not ffwf_outbound_status_to_wms_statuses.__contains__(ffwf_status):
                        continue
                    correct_wms_statuses = ffwf_outbound_status_to_wms_statuses[ffwf_status]
                    is_correct = False
                    for correct_wms_status in correct_wms_statuses:
                        if wms_status == correct_wms_status:
                            is_correct = True
                    if not is_correct:
                        wms_date_string = wms_order_statuses[service_request_id][1]
                        parsed_date = None
                        if wms_date_string == 'None':
                            wms_date_string = '-'
                        else:
                            parsed_date = self.parse_date(wms_date_string)
                        now = datetime.now()
                        if parsed_date is None or parsed_date + timedelta(hours=7) < now:
                            problems_map[ffwf_id] = ('Изъятие (' + str(type) + '/' + str(subtype) + '): ' + ffwf_id + '/' + service_request_id + '. FF: ' +
                                                     ffwf_status + ' (' + ffwf_date + ') WMS: ' + wms_status + ' (' + wms_date_string + ')\n\n')
        return problems_map

    def create_wms_connection(self, service_id):
        user = sdk2.Vault.data(self.get_mapping(self.Parameters.database_users_map_string)[str(service_id)][0])
        password = sdk2.Vault.data(self.get_mapping(self.Parameters.database_passwords_map_string)[str(service_id)][0])
        host = self.get_mapping(self.Parameters.database_hosts_map_string)[str(service_id)][0]
        port_mapping = self.get_mapping(self.Parameters.database_ports_map_string)
        if not port_mapping.__contains__(str(service_id)):
            port = '1433'
        else:
            port = port_mapping[str(service_id)][0]
        logging.info('Connecting to WMS, warehouse ' + str(service_id))
        _pymssql = __import__('pymssql')
        conn = _pymssql.connect(user=user,
                               password=password,
                               host=host,
                               port=port)
        logging.info('Connected')
        return conn

    def get_ffwf_inbound_status_to_wms_status(self):
        return self.get_mapping(str(self.Parameters.inbound_status_mappings))

    def get_ffwf_outbound_status_to_wms_status(self):
        return self.get_mapping(str(self.Parameters.outbound_status_mappings))

    def get_mapping(self, mappings_string):
        if not mappings_string:
            return {}
        result = {}
        parts = mappings_string.split(';')
        for part in parts:
            trimmed_part = part[1:-1].strip()
            key_and_value = trimmed_part.split(',')
            trimmed_key = key_and_value[0].strip()
            trimmed_value = key_and_value[1].strip()
            if not result.__contains__(trimmed_key):
                result[trimmed_key] = []
            result[trimmed_key].append(trimmed_value)
        return result

    def get_inbound_types(self):
        result = []
        parts = self.Parameters.inbound_types.split(';')
        for part in parts:
            trimmed_part = part.strip()
            result.append(trimmed_part)
        return set(result)

    def get_outbound_types(self):
        result = []
        parts = self.Parameters.outbound_types.split(';')
        for part in parts:
            trimmed_part = part.strip()
            result.append(trimmed_part)
        return set(result)

    def get_inbound_types_to_check_trailer(self):
        result = []
        parts = self.Parameters.inbound_types_to_check_trailer.split(';')
        for part in parts:
            trimmed_part = part.strip()
            result.append(trimmed_part)
        return set(result)

    def parse_date(self, date_string):
        try:
            return datetime.strptime(date_string, '%Y-%m-%d %H:%M:%S.%f')
        except Exception:
            return datetime.strptime(date_string, '%Y-%m-%d %H:%M:%S')

    def delete_leading_zeros(self, status):
        if status == '0':
            return status
        i = 0
        while status[i] == '0':
            if status[i:] == '0':
                return status[i:]
            i += 1
        return status[i:]

    def create_or_update_or_close_tickets(self, problems, service_id):
        from startrek_client import Startrek
        st = Startrek(useragent=self.Parameters.st_user_agent,
                      base_url=self.Parameters.st_url,
                      token=sdk2.Vault.data(self.Parameters.st_token_vault_key))

        tag = str(self.get_mapping(str(self.Parameters.tag_for_warehouse))[service_id])[2:-2]

        issues = st.issues.find('Queue: ' + str(self.Parameters.st_queue) +
                                ' Status: !Archived' +
                                ' Tags: ' + tag +
                                ' "Sort By": Key DESC',
                                per_page=1000
                                )
        updated_ids = []
        if issues:
            for issue in issues:
                summary = issue.summary
                ffwf_id = None
                try:
                    ffwf_id = summary[(len(str(self.Parameters.st_summary).decode('utf-8')) + 3):]
                except Exception as error:
                    logging.info(error)
                    continue
                if ffwf_id is None or ffwf_id == '':
                    continue
                if not problems.__contains__(ffwf_id):
                    self.close_non_actual_ticket(issue)
                else:
                    issue.comments.create(text='Проблема все еще актуальна')
                    updated_ids.append(ffwf_id)

        for ffwf_id, problem in problems.items():
            if updated_ids.__contains__(ffwf_id):
                continue
            st.issues.create(
                queue=str(self.Parameters.st_queue),
                summary=str(self.Parameters.st_summary) + ' - ' + ffwf_id,
                type={'name': 'Technical Task'},
                description=problem,
                tags=[tag]
            )

    def close_non_actual_ticket(self, issue):
        transition = issue.transitions['archived']
        transition.execute(comment='Проблема более не актуальна')

    def on_execute(self):
        bot = TelegramBot(bot_token=sdk2.Vault.data(self.Parameters.telegram_bot_token_vault_key))
        ffwf_requests = self.ffwfRequests()
        telegram_message = ''
        board_for_warehouse = self.get_mapping(str(self.Parameters.board_for_warehouse))
        monitoring_warehouses = []
        parts = self.Parameters.warehouses_for_monitoring.split(',')
        for part in parts:
            trimmed_part = part.strip()
            monitoring_warehouses.append(trimmed_part)
        for service_id in sorted(monitoring_warehouses):
            if not ffwf_requests.__contains__(int(service_id)):
                telegram_message += 'Отсутствуют расхождения по статусам заявок в FFWF и WMS для склада ' + str(service_id) + '\n\n'
                continue
            requests_info = ffwf_requests[int(service_id)]
            wms_statuses = self.checkWmsStatus(service_id, requests_info)
            inbounds_having_closed_trailers = self.inbounds_having_closed_trailers(requests_info, wms_statuses, service_id)
            warehouse_problems = self.compare_statuses(requests_info, wms_statuses, inbounds_having_closed_trailers)
            service_id_str = str(service_id)
            if len(warehouse_problems) == 0:
                telegram_message += 'Отсутствуют расхождения по статусам заявок в FFWF и WMS для склада ' + service_id_str + '\n\n'
                logging.info(telegram_message)
            else:
                telegram_message += 'Найдено ' + str(len(warehouse_problems)) + \
                                    ' расхождений по статусам заявок в FFWF и WMS для склада ' + service_id_str + \
                                    ', подробности - ' + str(board_for_warehouse[str(service_id)])[2:-2] + '\n\n'
            self.create_or_update_or_close_tickets(warehouse_problems, str(service_id))
        bot.send_message(self.Parameters.telegram_chat_id, telegram_message)
