# coding=utf-8

import json
import logging
import time
import re
from sandbox import sdk2
from sandbox import common
from sandbox.common.telegram import TelegramBot
from sandbox.sandboxsdk import environments

import requests


class IncorrectlyCisOnReport(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        telegram_notification = sdk2.parameters.Bool("Send report to Telegram", default=False)
        with telegram_notification.value[True]:
            telegram_chat_ids = sdk2.parameters.String('Telegram chat IDs in format (chat_id, chat_id)', required=True)
            telegram_bot_token_vault = sdk2.parameters.Vault("Vault secret contains Telegram bot token", required=True)

        solomon_api_url = sdk2.parameters.String('Solomon api url', default='http://solomon.yandex.net')
        project_id = sdk2.parameters.String('Project id')
        service_name = sdk2.parameters.String('Service name')
        cluster_name = sdk2.parameters.String('Cluster name')
        oauth_token_vault_key = sdk2.parameters.String('Oauth token vault key',
                                                       default='market_ss_solomon_oauth_token')
        sensor_label = sdk2.parameters.String('Sensor label')

        st_user_agent = sdk2.parameters.String('Startrack user agent', required=True)
        st_url = sdk2.parameters.String('Startrack url', required=True)
        st_token_vault_key = sdk2.parameters.String('Startrack token vault key', required=True)
        st_queue = sdk2.parameters.String('Startrack queue', required=True)
        st_summary = sdk2.parameters.String('Startrack ticket summary', required=True)
        st_assignee = sdk2.parameters.String('Startrack assignee', required=True)
        st_assignees_mapping_for_warehouses = sdk2.parameters.String(
            'Startrack assignee for databases:String in format (warehouseId, assignee)[;(warehouseId, assignee)]',
            required=False)
        st_followers = sdk2.parameters.String('Startrack followers', required=True)
        st_maillistSummonees_mapping_for_warehouses = sdk2.parameters.String(
            'Startrack maillistSummonees for databases:String in format (warehouseId, mail)[;(warehouseId, mail)]',
            required=False)
        mapping_for_warehouses = sdk2.parameters.String(
            'Database hosts:String in format (warehouseId, databaseHost)[;(warehouseId, databaseHost)]', required=True)
        failed_without_check_warehouse = sdk2.parameters.String(
            'Warehouse_id which need to be failed if not present in format (warehouseId, warehouseId)', required=True)

        yql_token_name = sdk2.parameters.String(
            "YQL token name",
            default='YQL_TOKEN',
            required=True
        )

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5
        environments = (environments.PipEnvironment('psycopg2-binary'),
                        environments.PipEnvironment('yandex-yt'),
                        environments.PipEnvironment('yql'),
                        environments.PipEnvironment('yandex_tracker_client', version="1.3",
                                                    custom_parameters=["--upgrade-strategy only-if-needed"]),
                        environments.PipEnvironment('startrek_client', version="2.3.0",
                                                    custom_parameters=["--upgrade-strategy only-if-needed"]))

    def get_indexer_warehouseId_to_s_sku_and_supplierId_map(self, warehouses):
        import yql.api.v1.client
        from yql.client.operation import YqlOperationType

        result = {}

        if len(warehouses) == 0:
            return result

        yql_token = sdk2.Vault.data(self.Parameters.yql_token_name)
        yql_query = "USE chyt.arnold; SELECT warehouse_id, shop_sku, supplier_id FROM concatYtTablesRange(" \
                    "`//home/market/production/indexer/gibson/mi3/main/last_complete/genlog`) WHERE warehouse_id in (" \
                    + ','.join(map(str, warehouses)) + ") AND hasAny(cargo_types, [980]) and disabled = '0' "
        with yql.api.v1.client.YqlClient(token=yql_token) as yql_client:
            logging.info(yql_query)
            query = yql_client.query(yql_query)
            query.type = YqlOperationType.CLICKHOUSE
            query.run()

            if not query.get_results().is_success:
                raise common.errors.TaskFailure(
                    "YQL запрос не выполнен: " + '; '.join(str(error) for error in query.get_results().errors))

            for table in query.get_results():
                table.fetch_full_data()

                for row in table.rows:
                    if str(row[0]) in result:
                        result.get(str(row[0])).add(((unicode(row[1]).encode('UTF-8')) + "_" + str(row[2])))
                    else:
                        result[str(row[0])] = {((unicode(row[1]).encode('UTF-8')) + "_" + str(row[2]))}

                return result

    def get_indexer_warehouseId_to_s_sku_and_supplierId_map_by_market_objects(self, market_objects, warehouse):
        import yql.api.v1.client
        from yql.client.operation import YqlOperationType

        result = []

        if len(market_objects) == 0:
            return result

        manufacturer_and_storer = []

        for market_object in market_objects:
            manufacturer_and_storer.append(unicode("(\'" + market_object.manufacturer.decode('UTF-8') + "\', "
                + market_object.storer + ")").encode('UTF-8'))

        logging.info(manufacturer_and_storer)

        yql_token = sdk2.Vault.data(self.Parameters.yql_token_name)
        yql_query = "USE chyt.arnold; SELECT warehouse_id, shop_sku, supplier_id FROM concatYtTablesRange(" \
                    "`//home/market/production/indexer/gibson/mi3/main/last_complete/genlog`) WHERE warehouse_id = " \
                    "" + warehouse + " AND  (shop_sku, supplier_id) in (" + ','.join(
            map(str, manufacturer_and_storer)) + ") AND hasAny(cargo_types, [980]) and disabled = '0'"

        with yql.api.v1.client.YqlClient(token=yql_token) as yql_client:
            logging.info(yql_query)
            query = yql_client.query(unicode(yql_query.decode('UTF-8')))
            query.type = YqlOperationType.CLICKHOUSE
            query.run()

            if not query.get_results().is_success:
                raise common.errors.TaskFailure(
                    "YQL запрос не выполнен: " + '; '.join(str(error) for error in query.get_results().errors))

            for table in query.get_results():
                table.fetch_full_data()

                for row in table.rows:
                    shop_sku = (unicode(row[1]).encode('UTF-8'))
                    supplier_id = str(row[2])

                    for market_object in market_objects:
                        if shop_sku == market_object.manufacturer and supplier_id == market_object.storer:
                            result.append(shop_sku + "_" + supplier_id + "_" + market_object.serialnumber + "_"
                                + market_object.lot + "_" + market_object.loc)
                            break

                return set(result)

    def find_wms_stocks_without_cis(self, table):
        import yql.api.v1.client
        yql_token = sdk2.Vault.data(self.Parameters.yql_token_name)

        yql_query = "SELECT manufacturersku, storer, serialnumber, lot, loc FROM " \
                    "" + table + " WHERE type = 'CIS' and loc not regexp '^(LOST|PICKTO|INTRANSIT|T_TRANSIT|PACK|DROP|STAGE.*|DAMAGE.*|UPACK.*|S\\d+)$' and coalesce(hold, '0') != '1'"

        yql_test_query = "SELECT * FROM " + table + " LIMIT 1"

        with yql.api.v1.client.YqlClient(token=yql_token) as yql_client:
            logging.info(yql_test_query)
            query = yql_client.query(yql_test_query)
            query.run()
            if len(query.full_dataframe.values.tolist()) < 1:
                return []

            logging.info(yql_query)
            query = yql_client.query(yql_query)
            query.run()

            if not query.get_results().is_success:
                raise common.errors.TaskFailure(
                    "YQL запрос не выполнен: " + '; '.join(str(error) for error in query.get_results().errors))

            result = []
            for table in query.get_results():
                table.fetch_full_data()

                for row in table.rows:
                    result.append(MarketObject((unicode(row[0]).encode('UTF-8')), str(row[1]), str(row[2]), str(row[3]),
                        str(row[4])))

            return list(set(result))

    def get_mapping(self, mappings_string):
        result = {}
        parts = mappings_string.split(';')
        parts.remove('')
        for part in parts:
            striped_part = part.strip()
            trimmed_part = striped_part[1:-1].strip()
            key_and_value = trimmed_part.split(',')
            trimmed_key = key_and_value[0].strip()
            trimmed_value = key_and_value[1].strip()
            result[trimmed_key] = trimmed_value

        logging.info(result)
        return result

    def on_execute(self):
        warehouses = self.get_mapping(
            str(self.Parameters.mapping_for_warehouses))
        is_failing_without_db = eval(str(self.Parameters.failed_without_check_warehouse))
        for key in warehouses:
            found_in_warehouse = self.find_wms_stocks_without_cis(warehouses.get(key))
            sku_and_supplier_and_other_product_fields = self.get_indexer_warehouseId_to_s_sku_and_supplierId_map_by_market_objects(
                found_in_warehouse, key)
            self.create_report_in_startrack(sku_and_supplier_and_other_product_fields, key, True)

        found_in_indexer = self.get_indexer_warehouseId_to_s_sku_and_supplierId_map(is_failing_without_db)
        for key in found_in_indexer:
            self.create_report_in_startrack(found_in_indexer.get(key), key, False)

    def form_table_to_be_human_readable_format(self, actual_problems, supported_warehouses):
        table_string = "#|\n"

        if supported_warehouses == True:
            table_string = table_string + "|| shop_sku | supplier_id | УИТ | партия | ячейка|| \n"
        else:
            table_string = table_string + "|| shop_sku | supplier_id|| \n"

        for string in actual_problems:
            split = string.split('_')
            if supported_warehouses == True:
                table_string = table_string + "|| " + split[0] + " | " + split[1] + " | " + split[2] + \
                    " | " + split[3] + " | " + split[4] + " ||\n"
            else:
                table_string = table_string + "|| " + split[0] + " | " + split[1] + " ||\n"

        table_string = table_string + "|#\n \n"
        table_string = table_string + "<{task_details\n" + ' ;'.join(actual_problems) + "\n}>"
        return table_string

    def create_report_in_startrack(self, set, warehouse_id, supported_warehouses):
        if len(set) == 0:
            return
        from startrek_client import Startrek
        st = Startrek(useragent=self.Parameters.st_user_agent,
                      base_url=self.Parameters.st_url,
                      token=sdk2.Vault.data(self.Parameters.st_token_vault_key))

        tag = 'warehouse ' + str(warehouse_id)
        issues_opened = st.issues.find('Queue: ' + str(self.Parameters.st_queue) +
                                       ' Status: Open' +
                                       ' Tags: "' + tag + '"' +
                                       ' "Sort By": Key DESC',
                                       per_page=1000)
        closed_today = st.issues.find('Queue: ' + str(self.Parameters.st_queue) +
                                      ' Status: Resolved,Closed' +
                                      ' Tags: "' + tag + '"' +
                                      ' Updated: today() ' +
                                      ' "Sort By": Key DESC',
                                      per_page=1000)

        actual_problems = list(set)


        opened_issues_count = 0

        for id in issues_opened:
            desc = re.search('{(task_details\n)(.*?)\n}', id.description).group(2)
            for problem_id in desc.split(' ;'):
                opened_issues_count = opened_issues_count + 1
                if unicode(problem_id).encode('UTF-8') in actual_problems:
                    actual_problems.remove(unicode(problem_id).encode('UTF-8'))

        for id in closed_today:
            desc = re.search('{(task_details\n)(.*?)\n}', id.description).group(2)
            for problem_id in desc.split(' ;'):
                if unicode(problem_id).encode('UTF-8') in actual_problems:
                    actual_problems.remove(unicode(problem_id).encode('UTF-8'))

        opened_issues_count = opened_issues_count + len(actual_problems)

        oauth_token = sdk2.Vault.data(self.Parameters.oauth_token_vault_key)
        pusher = Pusher(solomon_api_url=self.Parameters.solomon_api_url,
                        oauth_token=oauth_token,
                        project_id=self.Parameters.project_id,
                        service_name=self.Parameters.service_name,
                        cluster_name=self.Parameters.cluster_name
                        )

        pusher.push_single_sensor(sensor_label=self.Parameters.sensor_label + '.' + warehouse_id,
                                  value=opened_issues_count)

        logging.info('Warehouse "{}" with actual_problems {}'.format(warehouse_id, actual_problems))

        if len(actual_problems) > 0:
            warehouse_id_to_assignee = self.get_mapping(str(self.Parameters.st_assignees_mapping_for_warehouses))
            created = st.issues.create(
                queue=str(self.Parameters.st_queue),
                assignee=str(self.Parameters.st_assignee) if warehouse_id not in warehouse_id_to_assignee else str(
                    warehouse_id_to_assignee.get(warehouse_id)),
                followers=str(self.Parameters.st_followers),
                summary=self.Parameters.st_summary.format(warehouse_id),
                type={'name': 'Technical Task'},
                description=self.form_table_to_be_human_readable_format(actual_problems, supported_warehouses),
                tags=[tag]
            )
            message = 'Найдены товары на выдаче на складе {}, но без КИЗ'.format(warehouse_id)
            message = message + '\n Ссылка на задачу ' + "https://st.yandex-team.ru/" + created.key

            warehouse_id_to_maillist_summonees = self.get_mapping(
                str(self.Parameters.st_maillistSummonees_mapping_for_warehouses))
            if str(warehouse_id) in warehouse_id_to_maillist_summonees:
                logging.info('task: {}, maillistSummonees: {}'.format(created.key,
                                str(warehouse_id_to_maillist_summonees.get(warehouse_id))))
                st.issues[created.key].comments.create(
                    text='',
                    attachmentIds=[],
                    summonees=[],
                    maillistSummonees=[str(warehouse_id_to_maillist_summonees.get(warehouse_id))]
                )

            if self.Parameters.telegram_notification:
                bot_token = sdk2.Vault.data(self.Parameters.telegram_bot_token_vault)
                telegram_chats_id = eval(str(self.Parameters.telegram_chat_ids))
                for telegram_chat_id in telegram_chats_id:
                    try:
                        bot = TelegramBot(bot_token)
                        bot.send_message(telegram_chat_id, message)
                    except Exception as e:
                        logging.warn("Telegram notification failed: {}".format(e.message))


class MarketObject(object):
    def __init__(self,
                 manufacturer,
                 storer, serialnumber,
                 lot, loc):

        self.manufacturer = manufacturer
        self.storer = storer
        self.serialnumber = serialnumber
        self.lot = lot
        self.loc = loc

    def __repr__(self):
        return unicode("(\'" + self.manufacturer.decode('UTF-8') + "\', " + self.storer + "\', "
         + self.serialnumber + "\', " + self.lot + "\', " +  self.loc + ")").encode('UTF-8')

    def __eq__(self, other):
        if not isinstance(other, MarketObject):
            # don't attempt to compare against unrelated types
            return NotImplemented

        return self.manufacturer == other.manufacturer and self.storer == other.storer and \
            self.serialnumber == other.serialnumber and self.lot == other.lot and self.loc == other.loc


class Pusher(object):
    def __init__(self,
                 solomon_api_url,
                 oauth_token,
                 project_id,
                 service_name,
                 cluster_name):
        self.solomon_api_url = solomon_api_url
        self.oauth_token = oauth_token
        self.project_id = project_id
        self.service_name = service_name
        self.cluster_name = cluster_name

    def push_single_sensor(self, sensor_label, value):
        logging.info('Pushing sensor "{}" with value {}'.format(sensor_label, value))
        logging.info('Project id "{}", service name "{}", cluster_name "{}"'.format(self.project_id, self.service_name,
                                                                                    self.cluster_name))
        body = self.__make_data(sensor_label, value)
        response = requests.post(
            self.solomon_api_url + '/api/v2/push?project={project}&service={service}&cluster={cluster}'.format(
                project=self.project_id, service=self.service_name, cluster=self.cluster_name
            ), headers=self.__make_headers(), data=body)

        if response.status_code != 200:
            logging.info('Solomon responded with {} {} {}'.format(response.status_code, response.reason,
                                                                  response.text))

    def __make_headers(self):
        return {
            'Authorization': 'OAuth ' + self.oauth_token,
            'Content-Type': 'application/json'
        }

    @staticmethod
    def __make_data(sensor_label, value):
        return json.dumps({
            "sensors": [{
                "labels": {"sensor": sensor_label, "period": "one_hour"},
                "ts": int(time.time()),
                "value": value
            }]
        })
